#include <backend/backend.h>

#include "imap_command.h"
#include <common/quoted_string.h>
#include <common/settings.h>

#include <common/helpers/utf7imap.h>
#include <yplatform/yield.h>

#include <regex>

namespace yimap {

struct Select : ImapCommand
{
    using YieldCtx = yplatform::yield_context<Select>;

    bool readOnly;
    string folderName;
    FolderPtr folder;

    Select(ImapCommandArgs& args, bool readOnly)
        : ImapCommand(args, SessionState::AUTH), readOnly(readOnly)
    {
    }

    void exec() override
    {
        yplatform::spawn(ioService(), yplatform::shared_from(this));
    }

    void operator()(YieldCtx yieldCtx)
    {
        reenter(yieldCtx)
        {
            folderName = quotedArg(1);

            unselectCurrentFolder();

            if (!checkEncoding(folderName))
            {
                return completeEncodingError();
            }

            if (!hasFolderInCache(folderName))
            {
                yield updateFolderList().then(yieldCtx);
            }

            if (!isFolderSelectable(folderName))
            {
                return completeNonselectableFolder();
            }

            yield getFolderFromBackend(folderName).then(yieldCtx.capture(folder));
            selectFolder(folder);
            completeOkWithFolderInfo(folder);
        }
    }

    void operator()(YieldCtx::exception_type exception)
    {
        try
        {
            std::rethrow_exception(exception);
        }
        catch (const NoSuchFolderError& e)
        {
            completeNoSuchFolder(e.what());
        }
        catch (const InvalidFolderNameError& e)
        {
            completeInvalidFolderName(e.what());
        }
        catch (const Utf7EncodingError& e)
        {
            completeEncodingError(e.what());
        }
        catch (const ReservedFolderNameError& e)
        {
            completeReservedFolderName(e.what());
        }
        catch (const std::exception& e)
        {
            completeWithException(e);
        }
        catch (...)
        {
            completeWithUnknownException();
        }
    }

    void unselectCurrentFolder()
    {
        imapContext->sessionState.unselect();
    }

    bool hasFolderInCache(const std::string& folderName)
    {
        return imapContext->foldersCache.hasFolder(folderName);
    }

    bool isFolderSelectable(const std::string& folderName)
    {
        auto folders = imapContext->foldersCache.getFolders();
        return folders->isFolderSelectable(folderName);
    }

    Future<FolderPtr> getFolderFromBackend(const std::string& folderName)
    {
        auto folders = imapContext->foldersCache.getFolders();
        auto dbFolderId = folders->getDBFolderId(folderName);
        return metaBackend->getFolder(dbFolderId);
    }

    void selectFolder(FolderPtr folder)
    {
        if (readOnly)
        {
            imapContext->sessionState.selectFolderReadOnly(folder);
        }
        else
        {
            imapContext->sessionState.selectFolder(folder);
        }
    }

    void completeEncodingError(const string& message = "")
    {
        if (message.size())
        {
            logError() << command() << " can`t select folder(Utf7EncodingError) '" << folderName
                       << "': " << message;
        }
        completeBad("[CLIENTBUG]", "Folder encoding error.");
    }

    void completeNoSuchFolder(const string& message)
    {
        logError() << " can`t select folder (NoSuchFolderError) '" << folderName
                   << "': " << message;
        completeNo("[CLIENTBUG]", "No such folder.");
    }

    void completeInvalidFolderName(const string& message)
    {
        logError() << command() << " can`t select folder(InvalidFolderNameError) '" << folderName
                   << "': " << message;
        completeBad("[CLIENTBUG]", "Bad folder name.");
    }

    void completeReservedFolderName(const string& message)
    {
        logError() << command() << " can`t select folder(ReservedFolderNameError) '" << folderName
                   << "': " << message;
        completeNo("[UNAVAILABLE]", "No such folder.");
    }

    void completeNonselectableFolder()
    {
        logError() << " can`t select folder '" << folderName << "': Nonselectable folder";
        completeBad("[CLIENTBUG]", "Nonselectable folder.");
    }

    void completeOkWithFolderInfo(FolderPtr folder)
    {
        sendClient() << generateResponse(folder);
        completeOk(readOnly ? "[READ-ONLY]" : "[READ-WRITE]");
    }

    string generateResponse(FolderPtr folder)
    {
        std::stringstream os;

        const auto& folderInfo = folder->folderInfo;
        os << "* FLAGS (\\Answered \\Seen \\Draft" << (folder->isShared() ? "" : " \\Deleted")
           << " $Forwarded)\r\n"
           << "* " << folderInfo.messageCount << " EXISTS\r\n"
           << "* " << folderInfo.recentCount << " RECENT\r\n";

        if (folderInfo.firstUnseen != 0)
        {
            os << "* OK [UNSEEN " << folderInfo.firstUnseen << "]\r\n";
        }

        os << "* OK [PERMANENTFLAGS (\\Answered \\Seen \\Draft \\Flagged"
           << (folder->isShared() ? "" : " \\Deleted") << " $Forwarded \\*)] Limited\r\n";

        os << "* OK [UIDNEXT " << folderInfo.uidNext << "] Ok\r\n"
           << "* OK [UIDVALIDITY " << folderInfo.uidValidity << "] Ok\r\n";

        return os.str();
    }
};

CommandPtr CommandSelect(ImapCommandArgs& args)
{
    return CommandPtr(new Select(args, false));
}
CommandPtr CommandExamine(ImapCommandArgs& args)
{
    return CommandPtr(new Select(args, true));
}
}
