#include <backend/backend.h>

#include "imap_command.h"
#include <common/settings.h>
#include <common/sequence_ranges.h>
#include <yplatform/yield.h>

namespace yimap {

struct ExpungeBase : ImapSelectedCommand
{
    FolderRef mailbox;

    ExpungeBase(ImapCommandArgs& cmdArgs) : ImapSelectedCommand(cmdArgs)
    {
    }

    FolderRef getSelectedFolder()
    {
        return imapContext->sessionState.selectedFolder;
    }

    bool canExpunge(FolderRef mailbox)
    {
        bool isShared = imapContext->foldersCache.getSharedStatusByFid(mailbox.info().fid);
        bool isReadOnly = mailbox.readOnly();
        return !isShared && !isReadOnly;
    }

    Future<void> expunge(FolderRef mailbox, UidMapPtr messages)
    {
        return metaBackend->expunge(mailbox, messages);
    }

    void completeNoBackendError(const string& message)
    {
        logError() << "expunge error: " << message;
        completeNo("[UNAVAILABLE]", "Backend error");
    }
};

struct Expunge : ExpungeBase
{
    using YieldCtx = yplatform::yield_context<Expunge>;

    UidMapPtr messages;

    Expunge(ImapCommandArgs& cmdArgs) : ExpungeBase(cmdArgs)
    {
    }

    void exec() override
    {
        yplatform::spawn(ioService(), yplatform::shared_from(this));
    }

    void operator()(YieldCtx yieldCtx)
    {
        reenter(yieldCtx)
        {
            mailbox = getSelectedFolder();

            if (canExpunge(mailbox))
            {
                yield messagesToDelete().then(yieldCtx.capture(messages));
                if (messages->size())
                {
                    yield expunge(mailbox, messages).then(yieldCtx);
                }
            }

            yield completeOkAndOutputDiff().then(yieldCtx);
        }
    }

    void operator()(YieldCtx::exception_type exception)
    {
        try
        {
            std::rethrow_exception(exception);
        }
        catch (const std::exception& e)
        {
            completeNoBackendError(e.what());
        }
    }

    Future<UidMapPtr> messagesToDelete()
    {
        seq_range range = mailbox.seqRange(true);
        return metaBackend->loadMessagesToDelete(mailbox, range);
    }
};

struct UidExpunge : ExpungeBase
{
    using YieldCtx = yplatform::yield_context<UidExpunge>;

    seq_range seq;
    UidMapPtr messages;

    UidExpunge(ImapCommandArgs& cmdArgs) : ExpungeBase(cmdArgs)
    {
    }

    void exec() override
    {
        yplatform::spawn(ioService(), yplatform::shared_from(this));
    }

    void operator()(YieldCtx yieldCtx)
    {
        reenter(yieldCtx)
        {
            mailbox = getSelectedFolder();

            yield updateFolderAndSendDiff().then(yieldCtx);
            seq = extractArgsFromAST();
            if (canExpunge(mailbox))
            {
                yield messagesToDelete(seq).then(yieldCtx.capture(messages));
                if (messages->size())
                {
                    yield expunge(mailbox, messages).then(yieldCtx);
                }
            }

            yield completeOkAndOutputDiff().then(yieldCtx);
        }
    }

    void operator()(YieldCtx::exception_type exception)
    {
        try
        {
            std::rethrow_exception(exception);
        }
        catch (const std::exception& e)
        {
            completeNoBackendError(e.what());
        }
    }

    seq_range extractArgsFromAST()
    {
        Trees& trees = tree->data.trees;
        Trees& cmdArgs = trees[0].children;
        Trees& uidCmdArgs = cmdArgs[1].children;
        const TreeNode& seq_node = uidCmdArgs[0];
        seq_range seq = mailbox.seqRange(true);
        parseSeqRange(seq_node, seq);
        return seq;
    }

    Future<UidMapPtr> messagesToDelete(const seq_range& seq)
    {
        return metaBackend->loadMessagesToDelete(mailbox, seq);
    }

    void completeStatusUpdateError()
    {
        completeNoBackendError("status update failure");
    }
};

CommandPtr CommandExpunge(ImapCommandArgs& commandArgs)
{
    return CommandPtr(new Expunge(commandArgs));
}

CommandPtr CommandUidExpunge(ImapCommandArgs& commandArgs)
{
    return CommandPtr(new UidExpunge(commandArgs));
}

}
