#include "fetch/attribute_fetcher.h"
#include "fetch/attribute.h"
#include "fetch/fetch_detail.h"
#include "imap_command.h"
#include <backend/user_journal_types.h>

#include <macs/user_journal.h>
#include <yplatform/util/sstream.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace yimap {

using macs::MimeParts;

struct FetchBase : ImapSelectedCommand
{
    using YieldCtx = yplatform::yield_context<FetchBase>;
    using ImapSelectedCommand::ImapSelectedCommand;

    FolderRef mailbox;
    seq_range seq;
    UidMapPtr messagesChunk;
    std::deque<FetchAttribute> atts;

    string err;
    UidMap::Iterator msgIt;
    std::size_t attIdx = 0;
    BodyMetadataByMid bodyMeta;
    std::vector<StringPtr> attValues;
    MessageVector fetchedMessages;
    UidMapPtr unseenMessages = std::make_shared<UidMap>();
    bool changeSeen = false;

    StringPtr responsesBuffer = std::make_shared<string>();
    size_t responsesChunkSize = 1000; // 1 kB
    Duration sendTime = {};
    size_t totalMessagesCount = 0;

    ~FetchBase()
    {
        updateStatsOnReleaseBuffer(responsesBuffer->size());
    }

    void exec() override
    {
        yplatform::spawn(ioService(), yplatform::shared_from(this));
    }

    void operator()(YieldCtx yieldCtx)
    {
        reenter(yieldCtx)
        {
            mailbox = selectedFolder();
            atts = parseFetchAttributes();

            if (uidMode())
            {
                yield updateFolderAndSendDiff().then(yieldCtx);
            }

            seq = getSeqRange();
            do
            {
                yield loadNextChunk().then(yieldCtx.capture(messagesChunk));
                if (messagesChunk->empty()) break;

                if (needBodyMetadata())
                {
                    yield loadBodyMetadata(*messagesChunk->toSmidList())
                        .then(yieldCtx.capture(bodyMeta));
                }

                fetchedMessages.clear();
                unseenMessages->clear();
                for (msgIt = messagesChunk->begin();
                     msgIt != messagesChunk->end() && networkSession->isOpen();
                     ++msgIt)
                {
                    resetAttValues();
                    changeSeen = needMarkSeen(*msgIt);
                    if (changeSeen) markAsSeen(*msgIt);
                    for (attIdx = 0; attIdx < atts.size(); ++attIdx)
                    {
                        yield fetchAttribute(
                            *msgIt, atts[attIdx], yieldCtx.capture(err, attValues[attIdx]));
                        if (err.size()) break;
                    }
                    if (err.empty() && changeSeen && !hasFlagsAtt())
                    {
                        attValues.push_back({});
                        yield fetchAttribute(
                            *msgIt,
                            FetchAttribute(lex_ids::FETCH_ATT_FLAGS),
                            yieldCtx.capture(err, attValues.back()));
                    }
                    if (err.size())
                    {
                        respondError(*msgIt, err);
                        continue;
                    }
                    respond(*msgIt, attValues);
                    fetchedMessages.emplace_back(*msgIt);
                }

                if (unseenMessages->size())
                {
                    yield updateFlags(unseenMessages, { "\\Seen" }, {}).then(yieldCtx);
                }

                if (!mailbox.readOnly())
                {
                    yield removeRecentFlag(fetchedMessages).then(yieldCtx);
                    if (readMessage())
                    {
                        reportRead(fetchedMessages);
                    }
                }

                flushResponses();
            } while (messagesChunk->size());

            if (uidMode())
            {
                yield updateFolderAndSendDiff().then(yieldCtx);
            }

            if (needDropFreshCounter())
            {
                yield dropFreshCounter().then(yieldCtx);
            }

            complete();
        }
    }

    void operator()(YieldCtx::exception_type exception)
    {
        try
        {
            std::rethrow_exception(exception);
        }
        catch (const std::exception& e)
        {
            completeWithException(e);
        }
    }

    FolderRef selectedFolder()
    {
        return imapContext->sessionState.selectedFolder;
    }

    seq_range getSeqRange()
    {
        auto seq = mailbox.seqRange(uidMode());
        parseSeqRange(getSeqNode(), seq);
        return seq;
    }

    std::deque<FetchAttribute> parseFetchAttributes()
    {
        static std::set<long> frontAttrs = { lex_ids::FETCH_ATT_UID };
        std::deque<FetchAttribute> ret;
        if (uidMode()) ret.push_back(FetchAttribute(lex_ids::FETCH_ATT_UID));
        auto& att_node = getArgsNode();
        for (auto& node : att_node.children)
        {
            auto atts = createAttributes(node);
            for (auto& att : atts)
            {
                auto it = std::find(ret.begin(), ret.end(), att);
                if (it != ret.end()) continue;
                if (frontAttrs.count(att.id))
                {
                    ret.push_front(att);
                }
                else
                {
                    ret.push_back(att);
                }
            }
        }
        return ret;
    }

    Future<UidMapPtr> loadNextChunk()
    {
        auto range = calculateRemainingRange(seq, messagesChunk);
        if (range.empty()) return makeFuture(std::make_shared<UidMap>());

        return metaBackend->loadMessagesChunk(mailbox, range).then([this](auto future) {
            auto chunk = future.get();
            totalMessagesCount += chunk->size();
            return chunk;
        });
    }
    seq_range calculateRemainingRange(const seq_range& fullRange, UidMapPtr lastChunk)
    {
        if (!lastChunk || lastChunk->empty()) return fullRange;
        auto lastId = fullRange.uidMode() ? lastChunk->rbegin()->uid : lastChunk->rbegin()->num;
        return trim_left_copy(fullRange, lastId);
    }

    // Need st_ids and mime parts.
    bool needBodyMetadata() const
    {
        auto&& attrs = getArgsNode().children;
        static std::set<long> partialAttrs = { lex_ids::FETCH_ATT_FLAGS,
                                               lex_ids::FETCH_ATT_UID,
                                               lex_ids::FETCH_ATT_INTERNALDATE };
        for (auto&& node : attrs)
        {
            auto lex = node.value.id().to_long();
            if (!partialAttrs.count(lex)) return true;
        }
        return false;
    }

    Future<BodyMetadataByMid> loadBodyMetadata(const SmidList& mids)
    {
        return metaBackend->loadBodyMetadata(mids);
    }

    void resetAttValues()
    {
        attValues.clear();
        attValues.resize(atts.size());
    }

    template <typename Handler>
    void fetchAttribute(const MessageData& msg, const FetchAttribute& att, Handler&& handler)
    {
        try
        {
            auto fetcher = createAttributeFetcher({ att, mbodyBackend, imapContext->settings });
            fetcher->fetch(msg, bodyMeta[msg.smid()], handler);
        }
        catch (const std::exception& e)
        {
            handler(e.what(), nullptr);
        }
    }

    bool hasFlagsAtt()
    {
        auto it = std::find(atts.begin(), atts.end(), FetchAttribute(lex_ids::FETCH_ATT_FLAGS));
        return it != atts.end();
    }

    bool readMessage()
    {
        auto&& attrs = getArgsNode().children;
        static std::set<long> readAtts = {
            lex_ids::FETCH_ATT_BODY,   lex_ids::FETCH_ATT_BODY_PEEK,
            lex_ids::FETCH_ATT_BINARY, lex_ids::FETCH_ATT_BINARY_PEEK,
            lex_ids::FETCH_ATT_RFC822, lex_ids::FETCH_ATT_RFC822_TEXT
        };
        for (auto&& node : attrs)
        {
            auto lex = node.value.id().to_long();
            if (readAtts.count(lex)) return true;
        }
        return false;
    }

    Future<void> removeRecentFlag(const MessageVector& messages)
    {
        auto recentMessages = std::make_shared<UidMap>();
        for (auto& message : messages)
        {
            if (message.has_flag(Flags::MSG_RECENT))
            {
                recentMessages->insert(message);
            }
        }
        return updateFlags(recentMessages, {}, { "\\Recent" });
    }

    bool needMarkSeen()
    {
        if (mailbox.readOnly()) return false;
        auto&& attrs = getArgsNode().children;
        static std::set<long> markSeenAttrs = { lex_ids::FETCH_ATT_BODY,
                                                lex_ids::FETCH_ATT_BINARY,
                                                lex_ids::FETCH_ATT_RFC822,
                                                lex_ids::FETCH_ATT_RFC822_TEXT };
        for (auto&& node : attrs)
        {
            auto lex = node.value.id().to_long();
            if (markSeenAttrs.count(lex)) return true;
        }
        return false;
    }

    bool needMarkSeen(const MessageData& msg)
    {
        return needMarkSeen() && !msg.flags.hasFlag("\\Seen"s);
    }

    Future<void> updateFlags(UidMapPtr messages, const Flags& addFlags, const Flags& delFlags)
    {
        if (messages->empty()) return makeFuture();
        return metaBackend->updateFlags(mailbox, messages, delFlags, addFlags);
    }

    void markAsSeen(const MessageData& msg)
    {
        unseenMessages->insert(msg);
        msg.flags.addFlags(std::vector<string>{ "\\Seen"s });
    }

    bool needDropFreshCounter()
    {
        return !mailbox.readOnly() && readMessage() && settings_->dropFreshCounter &&
            imapContext->foldersCache.getFolders()->isInbox(mailbox.info().fid);
    }

    Future<void> dropFreshCounter()
    {
        return metaBackend->dropFreshCounter();
    }

    void reportRead(const MessageVector& messages)
    {
        if (!journal) return;
        std::vector<std::string> mids;
        mids.resize(messages.size());
        for (auto& message : messages)
        {
            mids.emplace_back(message.smid());
        }
        journal->logOperation<user_journal_types::FetchBody>(
            user_journal::parameters::id::state(std::string()),
            user_journal::parameters::id::affected(0u),
            user_journal::parameters::id::mids(mids));
    }

    void updateStatsOnFillBuffer(size_t bytes)
    {
        imapContext->stats->fetch_buffer_size += bytes;
        imapContext->stats->fetch_bytes_cumulative += bytes;
    }

    void updateStatsOnReleaseBuffer(size_t bytes)
    {
        imapContext->stats->fetch_buffer_size -= bytes;
    }

    void respond(const MessageData& message, const std::vector<StringPtr>& attributes)
    {
        std::size_t prevBufferSize = responsesBuffer->size();
        yplatform::sstream stream(*responsesBuffer);
        stream << "* " << message.num << " FETCH (";
        bool first = true;
        for (auto& att : attributes)
        {
            if (!first)
            {
                stream << " ";
            }
            stream << *att;
            first = false;
        }
        stream << ")\r\n";
        updateStatsOnFillBuffer(responsesBuffer->size() - prevBufferSize);
        if (responsesBuffer->size() >= responsesChunkSize)
        {
            flushResponses();
        }
    }

    void flushResponses()
    {
        if (responsesBuffer->empty()) return;
        auto sendStartTime = Clock::now();
        sendClient() << *responsesBuffer;
        updateStatsOnReleaseBuffer(responsesBuffer->size());
        responsesBuffer->clear();
        sendTime += Clock::now() - sendStartTime;
    }

    void respondError(const MessageData& message, const string& err)
    {
        logError() << "fetch failure: message " << message.num << ", uid " << message.uid
                   << ", mid " << message.smid() << ", stid " << bodyMeta[message.smid()].stid
                   << ", error: " << err;

        sendClient() << "* NO [UNAVAILABLE] " << command() << " message " << message.num << " uid "
                     << message.uid << ": failure due to backend error\r\n";
    }

    void completeNoMessages()
    {
        if (!mailbox.empty()) logWarning() << "FETCH no messages: " << seq;
        completeOk("[CLIENTBUG]", "completed (no messages)");
    }

    void complete() override
    {
        if (totalMessagesCount == 0)
        {
            return completeNoMessages();
        }
        // Mail.app loops forever on "BAD" response, and thunderbird shows popup on "NO"
        // response and Evolution seems does not refresh message list on "NO" response.
        // So we have to answer "OK" only here.
        auto message = "Completed"s + (networkSession->isOpen() ? "."s : ", client disconnected."s);
        if (fetchedMessages.empty())
        {
            completeNo("[UNAVAILABLE]", message);
        }
        else
        {
            completeOk("", message);
        }
    }

    ExtraStatFields statExtra() const override
    {
        ostringstream ss;
        ss << " [fetch transfer:" << toString(sendTime) << "]";
        return ExtraStatFields{ std::pair{ "extra"s, ss.str() },
                                std::pair{ "messages_count"s, std::to_string(totalMessagesCount) },
                                std::pair{ "details_loaded"s,
                                           std::to_string(needBodyMetadata()) } };
    }

    virtual bool uidMode() const = 0;
    virtual const TreeNode& getArgsNode() const = 0;
    virtual const TreeNode& getSeqNode() const = 0;
};

struct Fetch : FetchBase
{
    using FetchBase::FetchBase;

    bool uidMode() const override
    {
        return false;
    }

    const TreeNode& getArgsNode() const override
    {
        return tree->data.trees[0].children[2];
    }

    const TreeNode& getSeqNode() const override
    {
        return tree->data.trees[0].children[1];
    }
};

struct UidFetch : FetchBase
{
    using FetchBase::FetchBase;

    bool uidMode() const override
    {
        return true;
    }

    const TreeNode& getArgsNode() const override
    {
        const auto& cargs = tree->data.trees[0].children;
        return cargs[1].children[1];
    }

    const TreeNode& getSeqNode() const override
    {
        const auto& cargs = tree->data.trees[0].children;
        return cargs[1].children[0];
    }

    string commandDump() const override
    {
        auto result = fullCommand();
        for (auto&& arg : fetchArgs())
        {
            result += " " + node_to_string(arg.value);
        }
        return result;
    }

    const std::vector<TreeNode>& fetchArgs() const
    {
        assert(tree->data.trees.size() > 0);
        assert(tree->data.trees[0].children.size() > 1);
        return tree->data.trees[0].children[1].children;
    }
};

CommandPtr CommandFetch(ImapCommandArgs& commandArgs)
{
    return CommandPtr(new Fetch(commandArgs));
}

CommandPtr CommandUidFetch(ImapCommandArgs& commandArgs)
{
    return CommandPtr(new UidFetch(commandArgs));
}

}
