#include "attribute.h"
#include <boost/algorithm/string.hpp>

namespace yimap {

template <typename T, typename H, typename EQ>
inline std::size_t hash_value(boost::unordered_set<T, H, EQ> const& s)
{
    H h;
    std::size_t seed = 0;
    for (T const& t : s)
    {
        boost::hash_combine(seed, h(t));
    }
    return seed;
}

bool iequal_to::operator()(string const& x, string const& y) const
{
    return boost::algorithm::iequals(x, y, std::locale());
}

std::size_t ihash::operator()(string const& x) const
{
    std::size_t seed = 0;
    std::locale locale;
    for (char const& c : x)
    {
        boost::hash_combine(seed, std::toupper(c, locale));
    }
    return seed;
}

std::size_t hash_value(FetchSection const& p)
{
    std::size_t seed = 0;
    boost::hash_combine(seed, p.id);
    boost::hash_combine(seed, p.part);
    boost::hash_combine(seed, p.headers);
    return seed;
}

bool operator==(FetchSection const& a, FetchSection const& b)
{
    return (a.id == b.id) && (a.part == b.part) && (a.headers == b.headers);
}

std::size_t hash_value(FetchAttribute const& p)
{
    std::size_t seed = 0;
    boost::hash_combine(seed, p.id);
    boost::hash_combine(seed, p.section);
    boost::hash_combine(seed, p.range_start);
    boost::hash_combine(seed, p.range_size);
    return seed;
}

bool operator==(FetchAttribute const& a, FetchAttribute const& b)
{
    return (a.id == b.id) && (a.section == b.section) && (a.range_start == b.range_start) &&
        (a.range_size == b.range_size);
}

FetchAttribute createBodyAttribute(const TreeNode& node)
{
    auto fetchAttLex = static_cast<lex_ids::ids>(node.value.id().to_long());
    if (node.children.empty()) throw std::runtime_error("bad BODY or BODY.PEEK request");
    FetchAttribute fa(fetchAttLex);
    auto& fs = fa.section;
    auto& section = node.children[0];
    fs.id = static_cast<lex_ids::ids>(section.value.id().to_long());
    // find section part if present
    for (auto& n : section.children)
    {
        if (n.value.id().to_long() == lex_ids::SECTION_PART)
        {
            fs.part = string(n.value.begin(), n.value.end());
        }
    }
    // find headers, etc
    switch (section.value.id().to_long())
    {
    case lex_ids::SECTION_MSG_HEADER_FIELDS:
    case lex_ids::SECTION_MSG_HEADER_FIELDS_NOT:
        for (auto& n : section.children)
        {
            if (n.value.id().to_long() == lex_ids::HEADER_FLD_NAME)
                fs.headers.insert(string(n.value.begin(), n.value.end()));
        }
        break;
    case lex_ids::SECTION:
    case lex_ids::SECTION_MSG_HEADER:
    case lex_ids::SECTION_MSG_TEXT:
    case lex_ids::SECTION_TEXT_MIME:
        break;
    case lex_ids::SECTION_PART:
        fs.part = string(section.value.begin(), section.value.end());
        fs.id = lex_ids::SECTION_PART;
        break;
    default:
        throw std::runtime_error("unknown BODY[.PEEK] section");
    }
    if (node.children.size() > 1)
    {
        if (node.children[1].value.id().to_long() != lex_ids::SECTION_RANGE)
        {
            throw std::runtime_error("bad BODY[.PEEK] request: section.range expected");
        }
        auto& range = node.children[1];
        if (range.children.size() != 2)
        {
            throw std::runtime_error("bad BODY[.PEEK] request: section.range: 2 numbers expected");
        }
        try
        {
            fa.range_start = boost::lexical_cast<std::size_t>(
                string(range.children[0].value.begin(), range.children[0].value.end()));
            fa.range_size = boost::lexical_cast<std::size_t>(
                string(range.children[1].value.begin(), range.children[1].value.end()));
        }
        catch (...)
        {
            throw std::runtime_error("cannot convert section.range");
        }
    }
    return fa;
}

std::vector<FetchAttribute> createAttributes(const TreeNode& node)
{
    auto fetchAttLex = static_cast<lex_ids::ids>(node.value.id().to_long());
    switch (fetchAttLex)
    {
    case lex_ids::FETCH_ATT_ENVELOPE:
    case lex_ids::FETCH_ATT_BODY_STRUCTURE:
    case lex_ids::FETCH_ATT_BODY_STRUCTURE_NONEXT:
    case lex_ids::FETCH_ATT_XMID:
    case lex_ids::FETCH_ATT_XSTID:
    case lex_ids::FETCH_ATT_INTERNALDATE:
    case lex_ids::FETCH_ATT_FLAGS:
    case lex_ids::FETCH_ATT_RFC822_SIZE:
    case lex_ids::FETCH_ATT_RFC822_HEADER:
    case lex_ids::FETCH_ATT_RFC822_TEXT:
    case lex_ids::FETCH_ATT_RFC822:
    case lex_ids::FETCH_ATT_UID:
        return { FetchAttribute(fetchAttLex) };
    case lex_ids::FETCH_ATT_BINARY:
    case lex_ids::FETCH_ATT_BINARY_PEEK:
    case lex_ids::FETCH_ATT_BINARY_SIZE:
    case lex_ids::FETCH_ATT_BODY:
    case lex_ids::FETCH_ATT_BODY_PEEK:
        return { createBodyAttribute(node) };
    case lex_ids::FETCH_MACRO_FAST:
        return { FetchAttribute(lex_ids::FETCH_ATT_FLAGS),
                 FetchAttribute(lex_ids::FETCH_ATT_INTERNALDATE),
                 FetchAttribute(lex_ids::FETCH_ATT_RFC822_SIZE) };
    case lex_ids::FETCH_MACRO_ALL:
        return { FetchAttribute(lex_ids::FETCH_ATT_ENVELOPE),
                 FetchAttribute(lex_ids::FETCH_ATT_FLAGS),
                 FetchAttribute(lex_ids::FETCH_ATT_INTERNALDATE),
                 FetchAttribute(lex_ids::FETCH_ATT_RFC822_SIZE) };
    case lex_ids::FETCH_MACRO_FULL:
        return { FetchAttribute(lex_ids::FETCH_ATT_BODY_STRUCTURE_NONEXT),
                 FetchAttribute(lex_ids::FETCH_ATT_ENVELOPE),
                 FetchAttribute(lex_ids::FETCH_ATT_FLAGS),
                 FetchAttribute(lex_ids::FETCH_ATT_INTERNALDATE),
                 FetchAttribute(lex_ids::FETCH_ATT_RFC822_SIZE) };
    default:
        throw std::runtime_error("unknown FETCH_ATT from client");
    }
}

}
