#pragma once

#include "fetch_detail.h"
#include "attribute_fetcher.h"

#include <parser/rfc822/rfc822.h>
#include <backend/envelope.h>
#include <backend/literal_out.h>

#include <yplatform/encoding/base64.h>
#include <yplatform/encoding/quoted_printable.h>
#include <yplatform/util.h>

namespace yimap {

// Bug RTEC-3893
namespace detail {

using namespace yimap::rfc822::rfc2822;

struct FetchBinaryData
{
    typedef string::const_iterator iterator_t;
    typedef boost::iterator_range<iterator_t> data_t;
    typedef boost::optional<data_t> opt_data_t;
    typedef boost::optional<string> opt_string_t;

    opt_string_t codepage;
    opt_data_t body;
};

struct FetchBinaryActions : public rfc822::MessageReactor
{
    typedef string::const_iterator iterator_t;
    typedef rfc822::MessageReactor::data_range_t data_range_t;

    FetchBinaryData& data;
    explicit FetchBinaryActions(FetchBinaryData& data_) : data(data_)
    {
    }
    void on_field_data(const rfc822::FieldData& fd);
    void on_body(const data_range_t& body);
};

inline void FetchBinaryActions::on_field_data(const rfc822::FieldData& fd)
{
    if (yplatform::util::iequals(fd.name, "content-transfer-encoding"))
    {
        string contentTransferEncoding = backend::msg_trim_str(fd.value->raw);
        // http://support.microsoft.com/kb/975918/en-us
        // http://tools.ietf.org/html/rfc2045#section-6.1
        // According to RFC, content-transfer-encoding cannot be empty
        if (contentTransferEncoding.empty()) contentTransferEncoding = "7BIT";
        data.codepage.reset(contentTransferEncoding);
        return;
    }
}

inline void FetchBinaryActions::on_body(const data_range_t& body)
{
    data.body.reset(body);
}

bool decode_message_part(
    const string& src,
    string& dest,
    bool save_headers,
    const std::vector<std::string>& rawHeadersList)
{
    FetchBinaryData data;
    FetchBinaryActions actions(data);

    if (!rfc822::parseMessage(src, actions, rawHeadersList))
        throw std::runtime_error("cannot parse message envelope (baida failure?)");
    dest = "";
    if (save_headers)
    {
        if (data.body) dest.append(src.begin(), data.body.get().begin());
        else
            dest = src;
    }
    if (!data.body) return true;
    if (data.codepage)
    {
        if (yplatform::util::iequals(data.codepage.get(), "base64"))
        {
            yplatform::decoder dec;
            dest.reserve(src.size());
            std::back_insert_iterator<string> iter(dest);
            dec.decode(data.body.get().begin(), data.body.get().end(), iter);
        }
        else if (yplatform::util::iequals(data.codepage.get(), "quoted-printable"))
        {
            dest += yplatform::quoted_printable_decode(data.body.get());
        }
        else
        {
            dest += data.body.get();
        }
    }
    else
    {
        dest += data.body.get();
    }
    return true;
}

}

struct BinaryFetcher : AttributeFetcher
{
    FetcherArgs args;

    BinaryFetcher(const FetcherArgs& args) : args(args)
    {
    }

    void fetch(const MessageData& message, const BodyMetadata& bodyMeta, const Handler& handler)
        override
    {
        auto onBody = std::bind(
            &BinaryFetcher::onBodyLoaded, yplatform::shared_from(this), p::_1, p::_2, handler);
        args.mbodyBackend->loadMessage(
            bodyMeta.stid, args.att.section.part, bodyMeta.mimeParts, onBody);
    }

    void onBodyLoaded(const string& err, StringPtr msg, const Handler& handler)
    {
        if (err.size()) return handler(err, {});
        std::stringstream stream;
        stream << "BINARY[" << args.att.section.part << "]";
        if (args.att.range_size) stream << '<' << args.att.range_start << '>';
        stream << " ";
        string decoded_message;
        bool saveHeaders = args.att.section.part.empty();
        detail::decode_message_part(
            *msg, decoded_message, saveHeaders, args.settings->rawHeadersList);
        detail::send_msg(stream, decoded_message, args.att.range_start, args.att.range_size);
        handler("", std::make_shared<string>(stream.str()));
    }
};

struct BinaryTextFetcher : AttributeFetcher
{
    FetcherArgs args;

    BinaryTextFetcher(const FetcherArgs& args) : args(args)
    {
    }

    void fetch(const MessageData& message, const BodyMetadata& bodyMeta, const Handler& handler)
        override
    {
        args.mbodyBackend->loadBody(
            bodyMeta.stid,
            args.att.section.part,
            bodyMeta.mimeParts,
            std::bind(
                &BinaryTextFetcher::onBodyLoaded,
                yplatform::shared_from(this),
                p::_1,
                p::_2,
                handler));
    }

    void onBodyLoaded(const string& err, StringPtr msg, const Handler& handler)
    {
        if (err.size()) return handler(err, {});
        std::stringstream stream;
        stream << "BINARY[";
        if (args.att.section.part.size())
        {
            stream << args.att.section.part << ".";
        }
        stream << "TEXT]";
        if (args.att.range_size) stream << '<' << args.att.range_start << '>';
        stream << " ";
        detail::send_msg(stream, *msg, args.att.range_start, args.att.range_size);
        handler("", std::make_shared<string>(stream.str()));
    }
};

struct BinaryHeaderFieldsFetcher : AttributeFetcher
{
    FetcherArgs args;

    BinaryHeaderFieldsFetcher(const FetcherArgs& args) : args(args)
    {
    }

    void fetch(const MessageData& message, const BodyMetadata& bodyMeta, const Handler& handler)
        override
    {
        args.mbodyBackend->loadHeader(
            bodyMeta.stid,
            args.att.section.part,
            bodyMeta.mimeParts,
            std::bind(
                &BinaryHeaderFieldsFetcher::onHeadersLoaded,
                yplatform::shared_from(this),
                p::_1,
                p::_2,
                handler));
    }

    void onHeadersLoaded(const string& err, StringPtr headers, const Handler& handler)
    {
        if (err.size()) return handler(err, {});
        handler("", makeResponse(headers));
    }

    StringPtr makeResponse(StringPtr headers)
    {
        std::stringstream stream;
        stream << "BINARY[";
        if (args.att.section.part.size()) stream << args.att.section.part << '.';
        stream << "HEADER.FIELDS (" << boost::algorithm::join(args.att.section.headers, " ")
               << ")]";
        if (args.att.range_size) stream << '<' << args.att.range_start << '>';
        stream << ' ';
        auto filteredHeaders = detail::filtersByHeaders(
            *headers, args.att.section.headers, args.settings->rawHeadersList);
        detail::send_msg(stream, filteredHeaders, args.att.range_start, args.att.range_size);
        return std::make_shared<string>(stream.str());
    }
};

struct BinaryHeaderFieldsNotFetcher : AttributeFetcher
{
    FetcherArgs args;

    BinaryHeaderFieldsNotFetcher(const FetcherArgs& args) : args(args)
    {
    }

    void fetch(const MessageData& message, const BodyMetadata& bodyMeta, const Handler& handler)
        override
    {
        args.mbodyBackend->loadHeader(
            bodyMeta.stid,
            args.att.section.part,
            bodyMeta.mimeParts,
            std::bind(
                &BinaryHeaderFieldsNotFetcher::onHeadersLoaded,
                yplatform::shared_from(this),
                p::_1,
                p::_2,
                handler));
    }

    void onHeadersLoaded(const string& err, StringPtr headers, const Handler& handler)
    {
        if (err.size()) return handler(err, {});
        std::stringstream stream;
        stream << "BINARY[";
        if (args.att.section.part.size()) stream << args.att.section.part << '.';
        stream << "HEADER.FIELDS.NOT (" << boost::algorithm::join(args.att.section.headers, " ")
               << ")]";
        if (args.att.range_size) stream << '<' << args.att.range_start << '>';
        stream << ' ';
        auto filteredHeaders = detail::excludeHeaders(
            *headers, args.att.section.headers, args.settings->rawHeadersList);
        detail::send_msg(stream, filteredHeaders, args.att.range_start, args.att.range_size);
        handler("", std::make_shared<string>(stream.str()));
    }
};

struct BinaryMimeFetcher : AttributeFetcher
{
    FetcherArgs args;

    BinaryMimeFetcher(const FetcherArgs& args) : args(args)
    {
    }

    void fetch(const MessageData& message, const BodyMetadata& bodyMeta, const Handler& handler)
        override
    {
        args.mbodyBackend->loadHeader(
            bodyMeta.stid,
            args.att.section.part,
            bodyMeta.mimeParts,
            std::bind(
                &BinaryMimeFetcher::onHeadersLoaded,
                yplatform::shared_from(this),
                p::_1,
                p::_2,
                handler));
    }

    void onHeadersLoaded(const string& err, StringPtr headers, const Handler& handler)
    {
        if (err.size()) return handler(err, {});
        std::stringstream stream;
        stream << "BINARY[";
        if (args.att.section.part.size()) stream << args.att.section.part << '.';
        stream << "MIME]";
        if (args.att.range_size) stream << '<' << args.att.range_start << '>';
        stream << " ";
        detail::send_msg(stream, *headers, args.att.range_start, args.att.range_size);
        handler("", std::make_shared<string>(stream.str()));
    }
};

struct BinarySizeFetcher : AttributeFetcher
{
    FetcherArgs args;

    BinarySizeFetcher(const FetcherArgs& args) : args(args)
    {
    }

    void fetch(const MessageData& message, const BodyMetadata& bodyMeta, const Handler& handler)
        override
    {
        auto onBody = std::bind(
            &BinarySizeFetcher::onBodyLoaded, yplatform::shared_from(this), p::_1, p::_2, handler);
        args.mbodyBackend->loadMessage(
            bodyMeta.stid, args.att.section.part, bodyMeta.mimeParts, onBody);
    }

    void onBodyLoaded(const string& err, StringPtr msg, const Handler& handler)
    {
        if (err.size()) return handler(err, {});
        string decoded_message;
        bool saveHeaders = args.att.section.part.empty();
        detail::decode_message_part(
            *msg, decoded_message, saveHeaders, args.settings->rawHeadersList);
        std::stringstream stream;
        stream << "BINARY.SIZE[" << args.att.section.part << "] " << decoded_message.size();
        handler("", std::make_shared<string>(stream.str()));
    }
};

}
