#pragma once

#include "attribute_fetcher.h"
#include "fetch_detail.h"
#include <backend/envelope.h>

namespace yimap {

struct BodyFetcher : AttributeFetcher
{
    FetcherArgs args;

    BodyFetcher(const FetcherArgs& args) : args(args)
    {
    }

    void fetch(const MessageData& message, const BodyMetadata& bodyMeta, const Handler& handler)
        override
    {
        auto onBody = std::bind(
            &BodyFetcher::onBodyLoaded, yplatform::shared_from(this), p::_1, p::_2, handler);
        bool withHeader = args.att.section.part.empty();
        if (withHeader)
        {
            args.mbodyBackend->loadMessage(
                bodyMeta.stid, args.att.section.part, bodyMeta.mimeParts, onBody);
        }
        else
        {
            args.mbodyBackend->loadBody(
                bodyMeta.stid, args.att.section.part, bodyMeta.mimeParts, onBody);
        }
    }

    void onBodyLoaded(const string& err, StringPtr msg, const Handler& handler)
    {
        if (err.size()) return handler(err, {});
        std::stringstream stream;
        stream << "BODY[" << args.att.section.part << "]";
        if (args.att.range_size) stream << '<' << args.att.range_start << '>';
        stream << " ";
        detail::send_msg(stream, *msg, args.att.range_start, args.att.range_size);
        handler("", std::make_shared<string>(stream.str()));
    }
};

struct BodyTextFetcher : AttributeFetcher
{
    FetcherArgs args;

    BodyTextFetcher(const FetcherArgs& args) : args(args)
    {
    }

    void fetch(const MessageData& message, const BodyMetadata& bodyMeta, const Handler& handler)
        override
    {
        args.mbodyBackend->loadBody(
            bodyMeta.stid,
            args.att.section.part,
            bodyMeta.mimeParts,
            std::bind(
                &BodyTextFetcher::onBodyLoaded,
                yplatform::shared_from(this),
                p::_1,
                p::_2,
                handler));
    }

    void onBodyLoaded(const string& err, StringPtr body, const Handler& handler)
    {
        if (err.size()) return handler(err, {});
        std::stringstream stream;
        stream << "BODY[";
        if (args.att.section.part.size())
        {
            stream << args.att.section.part << ".";
        }
        stream << "TEXT]";
        if (args.att.range_size) stream << '<' << args.att.range_start << '>';
        stream << " ";
        detail::send_msg(stream, *body, args.att.range_start, args.att.range_size);
        handler("", std::make_shared<string>(stream.str()));
    }
};

struct RFC822BodyFetcher : AttributeFetcher
{
    FetcherArgs args;

    RFC822BodyFetcher(const FetcherArgs& args) : args(args)
    {
    }

    void fetch(const MessageData& message, const BodyMetadata& bodyMeta, const Handler& handler)
        override
    {
        args.mbodyBackend->loadMessage(
            bodyMeta.stid,
            args.att.section.part,
            bodyMeta.mimeParts,
            std::bind(
                &RFC822BodyFetcher::onBodyLoaded,
                yplatform::shared_from(this),
                p::_1,
                p::_2,
                handler));
    }

    void onBodyLoaded(const string& err, StringPtr body, const Handler& handler)
    {
        if (err.size()) return handler(err, {});
        std::stringstream stream;
        stream << "RFC822 ";
        detail::send_msg(stream, *body, args.att.range_start, args.att.range_size);
        handler("", std::make_shared<string>(stream.str()));
    }
};

struct RFC822BodyTextFetcher : AttributeFetcher
{
    FetcherArgs args;

    RFC822BodyTextFetcher(const FetcherArgs& args) : args(args)
    {
    }

    void fetch(const MessageData& message, const BodyMetadata& bodyMeta, const Handler& handler)
        override
    {
        args.mbodyBackend->loadBody(
            bodyMeta.stid,
            args.att.section.part,
            bodyMeta.mimeParts,
            std::bind(
                &RFC822BodyTextFetcher::onBodyLoaded,
                yplatform::shared_from(this),
                p::_1,
                p::_2,
                handler));
    }

    void onBodyLoaded(const string& err, StringPtr body, const Handler& handler)
    {
        if (err.size()) return handler(err, {});
        std::stringstream stream;
        stream << "RFC822.TEXT ";
        detail::send_msg(stream, *body, args.att.range_start, args.att.range_size);
        handler("", std::make_shared<string>(stream.str()));
    }
};

}
