#pragma once

#include "fetch_detail.h"
#include "../imap_command.h"
#include <backend/envelope.h>
#include <parser/rfc822/rfc822.h>
#include <common/uid_map.h>
#include <common/log/logger.h>
#include <common/sequence_ranges.h>
#include <common/convert_string.h>

#include <yplatform/convert/yconv.h>

#include <boost/range.hpp>
#include <boost/iterator/transform_iterator.hpp>
#include <boost/date_time/posix_time/posix_time_types.hpp>
#include <time.h>

#define MSG_CACHE 1

namespace yimap { namespace detail {

inline std::size_t ranged_length(std::size_t sz, FetchAttribute const& fa)
{
    if (!fa.range_size) return sz;
    if (fa.range_start > sz) return 0;
    if ((sz -= fa.range_start) > fa.range_size) return fa.range_size;
    return sz;
}

template <typename Sequence>
boost::iterator_range<typename Sequence::const_iterator> ranged_string(
    Sequence const& seq,
    FetchAttribute const& fa)
{
    if (!fa.range_size)
        return boost::iterator_range<typename Sequence::const_iterator>(seq.begin(), seq.end());

    typename Sequence::const_iterator start = seq.begin() + fa.range_start;
    typename Sequence::const_iterator end = start + ranged_length(seq.size(), fa);
    return boost::template make_iterator_range(start, end);
}

template <typename Stream>
void send_att(Stream& os, const string& seq, bool& sp, bool append_space = true)
{
    if (sp) os << ' ';
    os << seq;
    if (append_space) os << ' ';
    sp = true;
}

class FlagTransform
{
public:
    inline FlagTransform(bool& first_char) : first_char_(&first_char)
    {
        *first_char_ = true;
    }

    template <typename T>
    void operator()(T& v)
    {
        v = transformed_c(v);
    }

    char transformed_c(char sc) const
    {
        unsigned char& c = reinterpret_cast<unsigned char&>(sc);

        // (, ), {, SP, CTL, %, *, ], ", \,
        switch (c)
        {
        case '\\':
            if (is_first()) return c;

        case '\0':
        case '"':
        case '(':
        case ')':
        case '{':
        case ' ':
        case '%':
        case '*':
            // commented out because thunderbird does not escapes ]
            // case ']':
            return '_';

        default:
            break;
        }

        // CTL
        if (c <= 0x1f || c == 0x7f) return '_';

        return c;
    }

protected:
    inline bool is_first() const
    {
        if (*first_char_)
        {
            *first_char_ = false;
            return true;
        }
        return false;
    }

private:
    bool* first_char_;
};

template <typename Stream>
class EscapeFlag
{
public:
    EscapeFlag(Stream& strm) : strm_(strm), need_space_(false)
    {
    }

    void operator()(string const& flag)
    {
        std::string out;
        out.reserve(flag.size() * 2);

        try
        {
            out = flag;
            bool fc;
            std::for_each(out.begin(), out.end(), FlagTransform(fc));
        }
        catch (const std::exception& e)
        {
            throw std::runtime_error("transform flag error: "s + e.what());
        }

        if (!out.empty())
        {
            if (!need_space_) need_space_ = true;
            else
                strm_ << ' ';

            strm_ << out;
        }
    }

protected:
    static char transform(char c);

private:
    Stream& strm_;

    bool need_space_;
};

template <typename Stream>
void send_msg(
    Stream& stream,
    const string& data,
    std::size_t range_start = 0,
    std::size_t range_size = 0)
{
    if (data.empty()) return;
    range_start = std::min(range_start, data.length() - 1);
    if (range_size)
    {
        range_size = std::min(range_size, data.length() - range_start);
    }
    else
    {
        range_size = data.length() - range_start;
    }
    stream << '{' << range_size << "}\r\n" << data.substr(range_start, range_size);
}

inline string filtersByHeaders(
    const string& messageHeader,
    const HeadersSet& requiredHeaders,
    const std::vector<string>& rawHeadersList)
{
    string ret;
    rfc822::MessageDataReactor mdata; // XXX check errors
    rfc822::parseMessage(messageHeader, mdata, rawHeadersList);
    for (const rfc822::FieldData& header : mdata.header_list_data)
    {
        if (requiredHeaders.count(header.name))
        {
            ret += header.raw;
        }
    }
    ret += "\r\n";
    return ret;
}

inline string excludeHeaders(
    const string& messageHeader,
    const HeadersSet& exludeHeaders,
    const std::vector<string>& rawHeadersList)
{
    string ret;
    rfc822::MessageDataReactor mdata; // XXX check errors
    rfc822::parseMessage(messageHeader, mdata, rawHeadersList);
    for (const rfc822::FieldData& header : mdata.header_list_data)
    {
        if (!exludeHeaders.count(header.name))
        {
            ret += header.raw;
        }
    }
    ret += "\r\n";
    return ret;
}

}}
