#pragma once

#include "attribute_fetcher.h"
#include "fetch_detail.h"
#include <backend/envelope.h>
#include <backend/mbody/message_loader.h>
#include <boost/algorithm/string/join.hpp>

namespace yimap {

struct HeaderFetcherBase : AttributeFetcher
{
    FetcherArgs args;

    HeaderFetcherBase(const FetcherArgs& args) : args(args)
    {
    }

    void fetch(const MessageData& message, const BodyMetadata& bodyMeta, const Handler& handler)
        override
    {
        args.mbodyBackend->loadHeader(
            bodyMeta.stid,
            args.att.section.part,
            bodyMeta.mimeParts,
            std::bind(
                &HeaderFetcherBase::onHeadersLoaded,
                yplatform::shared_from(this),
                p::_1,
                p::_2,
                handler));
    }

    void onHeadersLoaded(const string& err, StringPtr headers, const Handler& handler)
    {
        if (err.size()) return handler(err, {});
        handler("", makeResponse(headers));
    }

    virtual StringPtr makeResponse(StringPtr headers) = 0;
};

struct HeaderFetcher : HeaderFetcherBase
{
    using HeaderFetcherBase::HeaderFetcherBase;

    StringPtr makeResponse(StringPtr headers) override
    {
        std::stringstream stream;
        stream << "BODY[";
        if (args.att.section.part.size()) stream << args.att.section.part << '.';
        stream << "HEADER]";
        if (args.att.range_size) stream << '<' << args.att.range_start << '>';
        stream << " ";
        detail::send_msg(stream, *headers, args.att.range_start, args.att.range_size);
        return std::make_shared<string>(stream.str());
    }
};

struct BinaryHeaderFetcher : HeaderFetcherBase
{
    using HeaderFetcherBase::HeaderFetcherBase;

    StringPtr makeResponse(StringPtr headers) override
    {
        std::stringstream stream;
        stream << "BINARY[";
        if (args.att.section.part.size()) stream << args.att.section.part << '.';
        stream << "HEADER]";
        if (args.att.range_size) stream << '<' << args.att.range_start << '>';
        stream << " ";
        detail::send_msg(stream, *headers, args.att.range_start, args.att.range_size);
        return std::make_shared<string>(stream.str());
    }
};

struct RFC822HeaderFetcher : HeaderFetcherBase
{
    using HeaderFetcherBase::HeaderFetcherBase;

    StringPtr makeResponse(StringPtr headers) override
    {
        std::stringstream stream;
        stream << "RFC822.HEADER ";
        detail::send_msg(stream, *headers);
        return std::make_shared<string>(stream.str());
    }
};

struct HeaderFieldsFetcher : HeaderFetcherBase
{
    using HeaderFetcherBase::HeaderFetcherBase;

    StringPtr makeResponse(StringPtr headers) override
    {
        std::stringstream stream;
        stream << "BODY[";
        if (args.att.section.part.size()) stream << args.att.section.part << '.';
        stream << "HEADER.FIELDS (" << boost::algorithm::join(args.att.section.headers, " ")
               << ")]";
        if (args.att.range_size) stream << '<' << args.att.range_start << '>';
        stream << ' ';
        auto filteredHeaders = detail::filtersByHeaders(
            *headers, args.att.section.headers, args.settings->rawHeadersList);
        detail::send_msg(stream, filteredHeaders, args.att.range_start, args.att.range_size);
        return std::make_shared<string>(stream.str());
    }
};

struct HeaderFieldsNotFetcher : HeaderFetcherBase
{
    using HeaderFetcherBase::HeaderFetcherBase;

    StringPtr makeResponse(StringPtr headers) override
    {
        std::stringstream stream;
        stream << "BODY[";
        if (args.att.section.part.size()) stream << args.att.section.part << '.';
        stream << "HEADER.FIELDS.NOT (" << boost::algorithm::join(args.att.section.headers, " ")
               << ")]";
        if (args.att.range_size) stream << '<' << args.att.range_start << '>';
        stream << ' ';
        auto filteredHeaders = detail::excludeHeaders(
            *headers, args.att.section.headers, args.settings->rawHeadersList);
        detail::send_msg(stream, filteredHeaders, args.att.range_start, args.att.range_size);
        return std::make_shared<string>(stream.str());
    }
};

}
