#pragma once

#include <backend/backend.h>
#include "imap_command.h"
#include <common/helpers/utf7imap.h>

namespace yimap {

struct FolderBase : ImapAuthenticatedCommand
{
    FolderBase(ImapCommandArgs& cmdArgs) : ImapAuthenticatedCommand(cmdArgs)
    {
    }

    bool isInbox(const string& folderName) const
    {
        return yplatform::iequals(folderName, "inbox");
    }

    bool isInboxChild(const string& folderName) const
    {
        return boost::istarts_with(folderName, "inbox|");
    }

    bool inboxChildrenDenied()
    {
        return !settings_->allowInboxSubfolders;
    }

    void completeBadNotAuthenticated()
    {
        completeBad("[CLIENTBUG]", "Not authenticated.");
    }

    void completeNoCantApplyToInbox()
    {
        logError() << "cannot apply " << command() << " to INBOX";
        completeNo("[CLIENTBUG]", "cannot apply to INBOX.");
    }

    void completeNoCantApplyToInboxChild()
    {
        logError() << "cannot apply " << command() << " to INBOX subfolder";
        completeNo("[CLIENTBUG]", "cannot apply to INBOX subfolder.");
    }

    void completeNoBackendError(const string& message = ""s)
    {
        if (message.size())
        {
            logError() << command() << " " << message;
        }
        completeNo("[UNAVAILABLE]", "Backend error.");
    }

    void completeBadFolderName(const string& message = ""s)
    {
        logError() << command() << " bad folder name: " << message;
        completeBad("[CLIENTBUG]", "Bad folder name.");
    }

    void completeNoReservedFolderName(const string& message = ""s)
    {
        logError() << command() << " reserved folder name: " << message;
        completeNo("[UNAVAILABLE]", "Reserved folder name.");
    }

    void completeBadFolderNameToolarge(const string& message = ""s)
    {
        if (message.size())
        {
            logError() << command() << " " << message;
        }
        completeBad("[CLIENTBUG]", "Folder name is too large.");
    }

    void completeBadEncodingError(const string& mbox, const string& message)
    {
        logError() << command() << " cannot convert from utf7 '" << mbox << "': " << message;
        completeBad("[CLIENTBUG]", "Folder encoding error.");
    }

    void completeNoSuchFolder(const string& mbox, const string& message)
    {
        logError() << command() << " no such folder " << mbox << " : " << message;
        completeNo("[CLIENTBUG]", "No such folder.");
    }
};

}
