#include "imap_command.h"
#include <backend/events.h>
#include <backend/handlers.h>
#include <common/settings.h>
#include "response/status_update.hpp"

namespace yimap {

static const string doneCmd = "done\r\n";

struct IdleImpl
    : ImapAuthenticatedCommand
    , ImapEventHandler
{
    IdleImpl(ImapCommandArgs& cmdArgs) : ImapAuthenticatedCommand(cmdArgs)
    {
    }

    void exec() override
    {
        assert(ioService().get_executor().running_in_this_thread());

        if (!idleExtensionEnabled())
        {
            completeBad("", "Command syntax error");
            return;
        }

        subscribeToEvents();
        sendContinuationRequest();
        readContinuationData();
    }

    bool idleExtensionEnabled() const
    {
        return settings_->idleExtension;
    }

    void subscribeToEvents()
    {
        notificationsBackend->subscribe(WeakEventHandlerPtr(weak_from(this)));
    }

    void readContinuationData()
    {
        networkSession->setIdleTimeouts();
        networkSession->asyncRead(doneCmd.size(), [this, self = shared_from(this)](ErrorCode ec) {
            networkSession->setDefaultTimeouts();

            if (isUnrecoverableError(ec))
            {
                return shutdown();
            }

            if (isCanceledError(ec))
            {
                return completeBye("Autologout; idle for too long");
            }

            if (startsWith(networkSession->readBuffer(), doneCmd))
            {
                networkSession->consumeReadBuffer(doneCmd.size());
                completeOk("", "Terminated");
            }
            else
            {
                completeBad("", "Command syntax error. ");
            }
        });
    }

    template <typename Range>
    bool startsWith(const Range& buffer, const string& prefix) const
    {
        return yplatform::util::istarts_with(buffer, prefix);
    }

    void shutdown()
    {
        networkSession->shutdown();
        promise->set(RET_CLOSE);
    }

    bool onEvent(ImapEventPtr event) override
    {
        // TODO don't create timer if milliseconds==0
        auto timer = createTimer(imapContext, Milliseconds(settings_->idleUpdateTimeout));
        timer->async_wait([this, weak = weak_from(this), event, timer](auto /*ec*/) {
            if (auto self = weak.lock())
            {
                // Output mailbox diff only in SELECTED state.
                if (!imapContext->sessionState.selected()) return;
                if (!event->checkConditions(imapContext)) return;
                updateFolderAndSendDiff();
            }
        });
        return true;
    }

    void sendContinuationRequest()
    {
        sendClient() << "+ idling\r\n";
    }

    bool isUnrecoverableError(ErrorCode ec) const
    {
        return ec && !isCanceledError(ec);
    }

    bool isCanceledError(ErrorCode ec) const
    {
        return ec == boost::asio::error::operation_aborted;
    }
};

CommandPtr CommandIdle(ImapCommandArgs& commandArgs)
{
    return CommandPtr(new IdleImpl(commandArgs));
}
}
