#include "imap_command.h"
#include "../module_context.h" // XXX
#include <common/settings.h>
#include <common/quoted_string.h>
#include <commands/response/status_update.hpp>

#include <common/helpers/utf7imap.h>

#include <common/log/logger.h>
#include <sstream>

namespace yimap {

ImapCommand::ImapCommand(ImapCommandArgs& cmdArgs, SessionState requiredState)
    : networkSession(cmdArgs.networkSession)
    , authBackend(cmdArgs.authBackend)
    , metaBackend(cmdArgs.metaBackend)
    , mbodyBackend(cmdArgs.mbodyBackend)
    , settingsBackend(cmdArgs.settingsBackend)
    , notificationsBackend(cmdArgs.notificationsBackend)
    , searchBackend(cmdArgs.searchBackend)
    , appendBackend(cmdArgs.appendBackend)
    , journal(cmdArgs.journal)
    , imapContext(cmdArgs.context)
    , settings_(cmdArgs.settings)
    , logger(cmdArgs.logger)
    , tree(cmdArgs.tree)
    , requiredState(requiredState)
{
    assert(logger);
    fillCommandData();
    promise = std::make_shared<ImapCommandPromise>();
    future = std::make_shared<ImapCommandFuture>(*promise);
}

void ImapCommand::fillCommandData()
{
    const TreeNode& cmdNode = tree->data.trees.front();
    tag_ = node_to_string(cmdNode.children[0].value);
    command_ = node_to_string(cmdNode.value);
    fullCommand_ = tag_ + " " + command_;

    if (boost::get<int>(cmdNode.value.value()) == CMD_UID)
    {
        fullCommand_ += " " + node_to_string(cmdNode.children[1].value);
        command_ += " " + node_to_string(cmdNode.children[1].value);
    }
}

string ImapCommand::commandDump() const
{
    string result = fullCommand_;
    for (size_t i = 1; i < cargsSize(); i++)
    {
        result += " ";
        result += carg(i);
    }
    return result;
}

void ImapCommand::init(StatsPtr aStats)
{
    stats = aStats;
    statistic = imapContext->getModuleContext();
}

void ImapCommand::start()
{
    startTime = Clock::now();
    if (!ioService().get_executor().running_in_this_thread())
    {
        throw std::runtime_error("command is running outside its IO service' thread");
    }

    try
    {
        if (imapContext->sessionState.checkAppropriateState(requiredState))
        {
            exec();
        }
        else
        {
            completeBad("[CLIENTBUG]", "Wrong session state for command");
        }
    }
    catch (...)
    {
        // TODO:
        //    TASK_LOG(imap_ctx, debug) << "exception in implementation::interpret call";
        promise->set_current_exception();
    }
}

Future<FolderInfo> ImapCommand::loadFolderInfo(const string& folderName)
{
    auto folders = imapContext->foldersCache.getFolders();
    if (!folders || !folders->hasFolder(folderName))
    {
        return updateFolderList().then([folderName, metaBackend = metaBackend](auto future) {
            auto folders = future.get();
            auto folderId = folders->getDBFolderId(folderName);
            return metaBackend->getFolderInfo(folderId);
        });
    }
    auto folderId = folders->getDBFolderId(folderName);
    return metaBackend->getFolderInfo(folderId);
}

Future<FolderListPtr> ImapCommand::updateFolderList()
{
    return metaBackend->getFolders().then(
        [imapContext = imapContext](auto&& future) -> Future<FolderListPtr> {
            FolderListPtr folders = future.get();
            Promise<FolderListPtr> promise;
            imapContext->ioService.post([promise, imapContext, folders]() mutable {
                imapContext->foldersCache.setFolders(folders);
                promise.set(folders);
            });
            return promise;
        });
}

Future<void> ImapCommand::updateFolderAndSendDiff()
{
    FolderRef mailbox = imapContext->sessionState.selectedFolder;
    if (!mailbox) return makeFuture();
    return metaBackend->statusUpdate(mailbox).then(
        [this, capture_self, mailbox](auto&& future) -> Future<void> {
            MailboxDiffPtr statusUpdate = future.get();
            Promise<void> promise;
            ioService().post([this, capture_self, mailbox, statusUpdate, promise]() mutable {
                checkMailboxConsistency(mailbox);
                sendStatusUpdate(statusUpdate);
                promise.set();
            });
            return promise;
        });
}

void ImapCommand::sendStatusUpdate(MailboxDiffPtr diff)
{
    if (diff && (diff->deleted.size() > 1 || (diff->deleted.size() == 1 && diff->changed.size())))
    {
        logWarning() << "respond status update: deleted=" << diff->deleted.size()
                     << ", changed=" << diff->changed.size();
    }
    sendClient() << StatusUpdateResponse::create(diff);
}

void ImapCommand::completeResponse(
    const string& result,
    const string& responseCode,
    const string& message,
    const string& errorSuffix)
{
    networkSession->clientStream()
        << tag() << " " << result << " " << responseCode << (!responseCode.empty() ? " " : "")
        << command() << " " << message << (errorSuffix.size() ? errorSuffix : ""s) << "\r\n";
}

Future<void> ImapCommand::completeOkAndOutputDiff(const string& respCode)
{
    return updateFolderAndSendDiff().then(
        [this, capture_self, respCode](auto&& future) -> Future<void> {
            future.get();
            Promise<void> promise;
            ioService().post([this, capture_self, respCode, promise]() mutable {
                completeOk(respCode);
                promise.set();
            });
            return promise;
        });
}

void ImapCommand::completeOk(const string& responseCode, const string& message)
{
    completeResponse("OK", responseCode, message);
    promise->set(RET_OK);
}

void ImapCommand::completeNo(const string& responseCode, const string& message)
{
    completeResponse("NO", responseCode, message, errorSuffix());
    promise->set(RET_NO);
}

void ImapCommand::completeBad(const string& responseCode, const string& message)
{
    completeResponse("BAD", responseCode, message, errorSuffix());
    promise->set(RET_BAD);
    reportBadCommand();
}

void ImapCommand::completeBye(const string& message)
{
    networkSession->clientStream() << "* BYE " << message << "\r\n";
    networkSession->shutdown();
    promise->set(RET_CLOSE);
}

void ImapCommand::completeByeAndOk(const string& byeMessage, const string& OkMessage)
{
    networkSession->clientStream() << "* BYE " << byeMessage << "\r\n";
    networkSession->clientStream() << tag() << " OK " << command() << " " << OkMessage << "\r\n";
    networkSession->shutdown();
    promise->set(RET_CLOSE);
}

void ImapCommand::completeWithException(const std::exception& e)
{
    logError() << command() << " exception: " << e.what();
    completeNo("[UNAVAILABLE]", "Internal server error.");
}

void ImapCommand::completeWithUnknownException()
{
    completeWithException(std::runtime_error("unknown"));
}

void ImapCommand::reportBadCommand()
{
    imapContext->processingLoopState.badCommands++;
}

string ImapCommand::stat() const
{
    ostringstream statsstream;
    auto time = Clock::now() - startTime;
    statsstream << "tm=" << toString(time);
    return statsstream.str();
}

// Length of tree->data.trees[0].children array
size_t ImapCommand::cargsSize() const
{
    return tree->data.trees[0].children.size();
}

// String value of tree->data.trees[0].children[index]
string ImapCommand::carg(size_t index) const
{
    return node_to_string(argNode(index).value);
}

string ImapCommand::quotedArg(size_t index) const
{
    return quoted_string(argNode(index).value);
}

string ImapCommand::quotedArgAsFolderName(size_t index) const
{
    auto range = quoted_range(argNode(index).value);
    if (!range.empty() && range.front() == '|') range.advance_begin(1);
    return boost::copy_range<string>(range);
}

void ImapCommand::checkMailboxConsistency(FolderRef folder)
{
    folder.checkConsistency(*logger);
}

// String from tree->data.trees.front().children[0].
// Asserts that exists at least 1 child. In fact c0 is tag.
string ImapCommand::c0() const
{
    Trees& c = tree->data.trees.front().children;
    assert(c.size() >= 1);
    return node_to_string(c[0].value);
}

// Unquoted string from tree->data.trees.front().children[1].
// Asserts that exists at least 2 children.
string ImapCommand::c1() const
{
    Trees& c = tree->data.trees.front().children;
    assert(c.size() >= 2);
    return quoted_string(c[1]);
}

// Unquoted string from tree->data.trees.front().children[2].
// Asserts that exists at least 3 children.
string ImapCommand::c2() const
{
    Trees& c = tree->data.trees.front().children;
    assert(c.size() >= 3);
    return quoted_string(c[2]);
}

bool ImapCommand::checkEncoding(const string& folderName)
{
    try
    {
        checkEncodingThrows(folderName);
    }
    catch (const Utf7EncodingError& e)
    {
        logError() << "cannot convert from utf7 '" << folderName << "': " << e.what();
        return false;
    }
    return true;
}

void ImapCommand::checkEncodingThrows(const string& folderName)
{
    folderNameFromUtf7Imap(folderName, '|');
}

}
