#pragma once

#include <common/types.h>
#include <common/log/logger.h>
#include <common/imap_context.h>
#include <common/settings.h>
#include <common/lex_ids.h>
#include <common/stats.h>

#include <parser/parser.h>
#include <common/tree.h>
#include <common/commands.h>
#include <backend/backend.h>
#include <backend/xiva/backend.h>
#include <network/session.h>
#include <yplatform/exception.h>
#include <yplatform/coroutine.h>

namespace yimap {

namespace backend {

class MetaBackend;
typedef std::shared_ptr<MetaBackend> MetaBackendPtr;

}

class ModuleContext;

using backend::MetaBackendPtr;

enum ReturnCode
{
    RET_OK,
    RET_NO,
    RET_BAD,
    RET_CLOSE,
};

typedef Promise<ReturnCode> ImapCommandPromise;
typedef std::shared_ptr<ImapCommandPromise> ImapCommandPromisePtr;

typedef Future<ReturnCode> ImapCommandFuture;
typedef std::shared_ptr<ImapCommandFuture> ImapCommandFuturePtr;

struct ImapCommandArgs
{
    ImapContextPtr context;
    NetworkSessionPtr networkSession;
    backend::AuthBackendPtr authBackend;
    backend::MetaBackendPtr metaBackend;
    backend::MbodyBackendPtr mbodyBackend;
    backend::UserSettingsBackendPtr settingsBackend;
    backend::NotificationsBackendPtr notificationsBackend;
    backend::SearchBackendPtr searchBackend;
    backend::AppendBackendPtr appendBackend;
    backend::UserJournalPtr journal;
    SettingsCPtr settings;
    Logger* logger;
    CommandASTPtr tree;
};

class ImapCommand : public std::enable_shared_from_this<ImapCommand>
{
protected:
    using SessionState = ImapContext::SessionState;
    using ExtraStatFields = map<string, string>;

public:
    ImapCommand(ImapCommandArgs& cmdArgs, SessionState requiredState = SessionState::INIT);
    virtual ~ImapCommand() = default;

    void init(StatsPtr stats);
    void start();

    ImapCommandFuturePtr getFuture() const
    {
        return future;
    }

    const string& tag() const
    {
        return tag_;
    }
    const string& command() const
    {
        return command_;
    }
    const string& fullCommand() const
    {
        return fullCommand_;
    }

    ClientStream sendClient()
    {
        return networkSession->clientStream();
    }

    LogHelper logError()
    {
        return logger->logError();
    }
    LogHelper logWarning()
    {
        return logger->logWarning();
    }
    LogHelper logDebug()
    {
        return logger->logDebug();
    }
    LogHelper logEvent()
    {
        return logger->logEvent();
    }

    const Settings& settings() const
    {
        return *settings_;
    }

    Duration totalTime() const
    {
        return Clock::now() - startTime;
    }

    virtual string stat() const;
    virtual ExtraStatFields statExtra() const
    {
        return {};
    }
    virtual string commandDump() const;

    IOService& ioService()
    {
        return imapContext->ioService;
    }

protected:
    virtual void exec() = 0;

    virtual void complete()
    {
    }

    Future<void> updateFolderAndSendDiff();
    Future<void> completeOkAndOutputDiff(const string& respCode = "");

    // Returns string like ". sc=LZivkqb04mI1" for error output ending.
    std::string errorSuffix() const
    {
        return createErrorSuffix(imapContext->uniq_id());
    }

    void fillCommandData();

    Future<FolderInfo> loadFolderInfo(const string& folderName);

    Future<FolderListPtr> updateFolderList();

    void completeOk(const string& responseCode = "", const string& message = "Completed.");
    void completeNo(const string& responseCode, const string& message);
    void completeBad(const string& responseCode, const string& message);
    void completeBye(const string& message);
    void completeByeAndOk(const string& byeMessage, const string& OkMessage = "Completed.");
    void completeWithException(const std::exception& e);
    void completeWithUnknownException();

    void reportBadCommand();

    MailboxDiffPtr loadStatusUpdate(FolderRef mailbox);
    void sendStatusUpdate(MailboxDiffPtr diff);

    // Length of args.tree->trees[0].children array
    size_t cargsSize() const;
    // String value of args.tree->trees[0].children[index]
    string carg(size_t index) const;
    string quotedArg(size_t index) const;
    string quotedArgAsFolderName(size_t index) const;
    auto argNode(size_t index) const
    {
        return tree->data.trees[0].children[index];
    }

    void checkMailboxConsistency(FolderRef folder);
    bool checkEncoding(const string& folderName);
    void checkEncodingThrows(const string& folderName);

private:
    // Output tagged command response as
    // "<tag> OK|NO|BAD <responseCode> <command_name> <mesaage> sc=<session_id>\r\n".
    // I.e. "a001 NO [UNAVAILABLE] FETCH failure due to backend error. sc=iRZhkvt33iE1".
    // Set OK promise.
    void completeResponse(
        const string& result,
        const string& responseCode,
        const string& message,
        const string& errorSuffix = ""s);

protected:
    // String from tree->trees.front().children[0]. In fact c0 is tag.
    // Asserts that exists at least 1 child.
    string c0() const;
    // Unquoted string from tree->trees.front().children[1].
    // Asserts that exists at least 2 children.
    string c1() const;
    // Unquoted string from tree->trees.front().children[2].
    // Asserts that exists at least 3 children.
    string c2() const;

protected:
    NetworkSessionPtr networkSession;
    backend::AuthBackendPtr authBackend;
    backend::MetaBackendPtr metaBackend;
    backend::MbodyBackendPtr mbodyBackend;
    backend::UserSettingsBackendPtr settingsBackend;
    backend::NotificationsBackendPtr notificationsBackend;
    backend::SearchBackendPtr searchBackend;
    backend::AppendBackendPtr appendBackend;
    backend::UserJournalPtr journal;

    ImapContextPtr imapContext;
    SettingsCPtr settings_;
    Logger* logger;

    CommandASTPtr tree;

    ImapCommandPromisePtr promise;
    ImapCommandFuturePtr future;

    StatsPtr stats;
    boost::shared_ptr<ModuleContext> statistic;

    string tag_;
    string command_;
    string fullCommand_;

    TimePoint startTime;

    SessionState requiredState;
};

typedef std::shared_ptr<ImapCommand> CommandPtr;
typedef std::weak_ptr<ImapCommand> WeakCommandPtr;

//------------------------------------------------------------------------------
// Authenticated state command

class ImapAuthenticatedCommand : public ImapCommand
{
public:
    ImapAuthenticatedCommand(ImapCommandArgs& cmdArgs) : ImapCommand(cmdArgs, SessionState::AUTH)
    {
    }
};

//------------------------------------------------------------------------------
// Selected state command

class ImapSelectedCommand : public ImapCommand
{
public:
    ImapSelectedCommand(ImapCommandArgs& cmdArgs) : ImapCommand(cmdArgs, SessionState::SELECT)
    {
    }
};

//------------------------------------------------------------------------------

template <typename NodeT>
string node_to_string(NodeT const& node)
{
    return string(node.begin(), node.end());
}

inline string toString(ReturnCode code)
{
    switch (code)
    {
    case ReturnCode::RET_OK:
        return "ok"s;
    case ReturnCode::RET_NO:
        return "no"s;
    case ReturnCode::RET_BAD:
        return "bad"s;
    case ReturnCode::RET_CLOSE:
        return "close"s;
    };
}

} // namespace yimap
