#include <algorithm>

#include "imap_command.h"
#include "list/wildcards.h"
#include <backend/backend.h>
#include <common/settings.h>
#include <common/quoted_string.h>
#include <common/convert_string.h>

#include <common/helpers/utf7imap.h>
#include <yplatform/yield.h>

static const char HIERARCHY_DELIM = '|';

namespace yimap {

struct ListBase : ImapCommand
{
    using NameToInfoMap = FolderList::Container;
    using YieldCtx = yplatform::yield_context<ListBase>;

    string referenceName;
    string folderWildcard;
    set<string> sharedNamespaces;
    NameToInfoMap folderByDisplayNames;

    ListBase(ImapCommandArgs& args) : ImapCommand(args, SessionState::AUTH)
    {
    }

    void exec() override
    {
        yplatform::spawn(ioService(), yplatform::shared_from(this));
    }

    void operator()(YieldCtx yieldCtx)
    {
        reenter(yieldCtx)
        {

            referenceName = getReferenceName();
            folderWildcard = getFolderWildcard();

            if (folderWildcard.empty())
            {
                return completeOkWithHierarchyEntry();
            }

            yield getAllFolders().then(yieldCtx.capture(folderByDisplayNames));
            sharedNamespaces = getSharedNamespaces();
            folderByDisplayNames = filter(folderByDisplayNames, referenceName, folderWildcard);
            completeOkWithFolders(folderByDisplayNames);
        }
    }

    void operator()(YieldCtx::exception_type exception)
    {
        try
        {
            std::rethrow_exception(exception);
        }
        catch (const Utf7EncodingError&)
        {
            completeEncodingError();
        }
        catch (const std::exception& e)
        {
            completeWithException(e);
        }
        catch (...)
        {
            completeWithUnknownException();
        }
    }

    string getReferenceName()
    {
        return folderNameFromUtf7Imap(quotedArg(1), HIERARCHY_DELIM);
    }

    string getFolderWildcard()
    {
        return folderNameFromUtf7Imap(quotedArg(2), HIERARCHY_DELIM);
    }

    Future<NameToInfoMap> getAllFolders()
    {
        return updateFolderList().then(
            [capture_self](auto future) { return future.get()->getInnerMap(); });
    }

    set<string> getSharedNamespaces()
    {
        return imapContext->foldersCache.getFolders()->getSharedNamespaces();
    }

    virtual NameToInfoMap filter(
        const NameToInfoMap& folders,
        const string& referenceName,
        const string& folderWildcard) = 0;

    NameToInfoMap filterSharedNamespaces(const NameToInfoMap& folders)
    {
        NameToInfoMap res;
        for (auto& pair : folders)
        {
            if (sharedNamespaces.find(pair.first) == sharedNamespaces.end())
            {
                res.insert(pair);
            }
        }
        return res;
    };

    NameToInfoMap filterByRefAndWildcard(
        const NameToInfoMap& folders,
        const string& referenceName,
        const string& folderWildcard)
    {
        NameToInfoMap res;
        for (auto& pair : folders)
        {
            auto utf8DisplayName = folderNameFromUtf7Imap(pair.first, HIERARCHY_DELIM);
            if (pair.second->symbol == "inbox" || boost::istarts_with(utf8DisplayName, "inbox|"))
            {
                if (!wildcard_match(
                        boost::to_lower_copy(referenceName),
                        boost::to_lower_copy(folderWildcard),
                        boost::to_lower_copy(utf8DisplayName),
                        HIERARCHY_DELIM))
                {
                    continue;
                }
            }
            else
            {
                if (!wildcard_match(
                        referenceName, folderWildcard, utf8DisplayName, HIERARCHY_DELIM))
                {
                    continue;
                }
            }
            res.insert(pair);
        }
        return res;
    }

    void completeOkWithHierarchyEntry()
    {
        sendClient() << makeUntaggedListResponse("\\Noselect", "");
        completeOk();
    }

    void completeOkWithFolders(const NameToInfoMap& folders)
    {
        sendClient() << makeUntaggedListResponses(folders);
        completeOk();
    }

    void completeEncodingError()
    {
        completeBad("[CLIENTBUG]", "Folder encoding error.");
    }

    virtual string makeUntaggedListResponses(const NameToInfoMap& folders)
    {
        string res;
        for (auto& [displayName, folder] : folders)
        {
            res += makeUntaggedListResponse(makeFolderFlags(folder), displayName);
        }
        return res;
    }

    string makeFolderFlags(FullFolderInfoPtr folder)
    {
        string flags = (folder->hasChildren ? "\\HasChildren" : "\\HasNoChildren");
        if (folder->symbol == "zombie")
        {
            flags += " \\Noselect";
            return flags;
        }

        flags += (folder->recentCount > 0 ? " \\Marked" : " \\Unmarked");

        if (folder->symbol == "inbox" && !settings_->allowInboxSubfolders)
        {
            flags += " \\NoInferiors";
        }

        auto specialUse = getFolderSpecialUseAttribute(folder);
        if (specialUse.size())
        {
            flags += " "s + specialUse;
        }
        return flags;
    }

    string makeUntaggedListResponse(const string& flags, const string& folderName)
    {
        return "* "s + command() + " ("s + flags + ") \""s + HIERARCHY_DELIM + "\" "s +
            (folderName.size() ? imap_convert_string(folderName) : "\"\"") + "\r\n"s;
    }

    string getFolderSpecialUseAttribute(FullFolderInfoPtr folder)
    {
        // Special use extension.
        string flag;
        if (folder->symbol == "sent")
        {
            flag = "\\Sent";
        }
        else if (folder->symbol == "spam")
        {
            flag = "\\Junk";
        }
        else if (folder->symbol == "trash")
        {
            flag = "\\Trash";
        }
        else if (folder->symbol == "draft")
        {
            flag = "\\Drafts";
        }
        else if (folder->symbol == "template")
        {
            flag = "\\Templates";
        }
        return flag;
    }
};

struct List : ListBase
{
    using ListBase::ListBase;

    NameToInfoMap filter(
        const NameToInfoMap& folders,
        const string& referenceName,
        const string& folderWildcard) override
    {
        auto filtered = filterSharedNamespaces(folders);
        return filterByRefAndWildcard(filtered, referenceName, folderWildcard);
    }
};

struct XList : List
{
    using List::List;

    string makeUntaggedListResponses(const NameToInfoMap& folders) override
    {
        string res;
        for (auto& [displayName, folder] : folders)
        {
            auto flags = makeFolderFlags(folder);
            auto xListFlag = getFolderXListFlag(folder);
            if (xListFlag.size())
            {
                flags += " "s + xListFlag;
            }
            res += makeUntaggedListResponse(flags, displayName);
        }
        return res;
    }

    string getFolderXListFlag(FullFolderInfoPtr folder)
    {
        auto folders = imapContext->foldersCache.getFolders();
        return folders->xlistFromSymbol(folder->symbol);
    }
};

struct LSub : ListBase
{
    using ListBase::ListBase;

    NameToInfoMap filter(
        const NameToInfoMap& folders,
        const string& referenceName,
        const string& folderWildcard) override
    {
        auto foldersWithoutNamespaces = filterSharedNamespaces(folders);
        auto unsubscribedFolders = filterUnsubscribed(foldersWithoutNamespaces);
        auto unsubscribedFoldersWithParents =
            addZombieParentsForParentlessFolders(unsubscribedFolders, foldersWithoutNamespaces);
        return filterByRefAndWildcard(
            unsubscribedFoldersWithParents, referenceName, folderWildcard);
    }

    NameToInfoMap filterUnsubscribed(const NameToInfoMap& folders)
    {
        NameToInfoMap res;
        for (auto& pair : folders)
        {
            if (pair.second->subscribed)
            {
                res.insert(pair);
            }
        }
        return res;
    }

    NameToInfoMap addZombieParentsForParentlessFolders(
        const NameToInfoMap& folders,
        const NameToInfoMap& allFolders)
    {
        NameToInfoMap res = folders;
        for (auto& pair : folders)
        {
            for (auto name = pair.first; name.size(); name = getParentName(name))
            {
                if (res.count(name)) continue;
                auto it = allFolders.find(name);
                if (it == allFolders.end()) break;

                it->second->symbol = "zombie";
                res.insert(*it);
            }
        }
        return res;
    }

    string getParentName(const string& folderName)
    {
        auto pos = folderName.rfind(HIERARCHY_DELIM);
        if (pos == string::npos) return "";
        return folderName.substr(0, pos);
    }
};

CommandPtr CommandList(ImapCommandArgs& args)
{
    return CommandPtr(new List(args));
}
CommandPtr CommandLsub(ImapCommandArgs& args)
{
    return CommandPtr(new LSub(args));
}
CommandPtr CommandXList(ImapCommandArgs& args)
{
    return CommandPtr(new XList(args));
}
}
