#include "wildcards.h"
#include <yplatform/log.h>
#include <boost/regex.hpp>
#include <iostream>

namespace yimap {

// returns true if imap_mailbox confirms mbox_ref and mbox_name
bool wildcard_match(
    const string& mbox_ref,
    const string& mbox_name,
    const string& imap_mailbox,
    char hdelim)
{
    // concatenate ref and name
    string name(mbox_ref);

    // trim ending delim symbols
    if (!hdelim)
    {
        name.append(mbox_name);
    }
    else
    {
        std::size_t pos;

        // Add delimiter, except first '%' symbol
        if ((name.size() > 0 && name[name.size() - 1] != hdelim) &&
            (mbox_name.empty() || mbox_name[0] != '%'))
        {
            name += hdelim;
        }
        pos = mbox_name.find_first_not_of(hdelim);
        if (pos != string::npos) name.append(mbox_name.substr(pos));
        else
            name.append(mbox_name);
    }

    // convert mbox_name into basic regular expression:
    // 1. escape . [ \ ^ $ chars
    // 2 convert * into .*
    // 3 convert % into [^<delim>]*
    string expr;
    expr.reserve(name.size() * 2);
    for (string::const_iterator iter = name.begin(); iter != name.end(); ++iter)
    {
        switch (*iter)
        {
        case '*':
            expr += ".*";
            break;

        case '%':
            if (hdelim)
            {
                expr += "[^";
                switch (hdelim)
                {
                case '.':
                case ']':
                case '\\':
                case '^':
                case '$':
                    expr += '\\';
                }
                expr += hdelim;
                expr += "]*";
            }
            else
                expr += ".*";
            break;

        case '.':
        case '[':
        case '\\':
        case '^':
        case '$':
            expr.push_back('\\');

        default:
            expr.push_back(*iter);
            break;
        }
    }

    // run regexp on imap_mailbox
    try
    {
        boost::regex pattern(
            expr,
            boost::regex::basic | boost::regex::nosubs | boost::regex::no_char_classes |
                boost::regex::no_intervals | boost::regex::no_except);
        return boost::regex_match(imap_mailbox, pattern);
    }
    catch (std::exception const& e)
    {
        string error = "wildcard_match: std exception in regexp_match (" + imap_mailbox + ", '" +
            expr + "'): " + e.what();
        throw std::runtime_error(error);
    }
    catch (...)
    {
        string error = "wildcard_match: unknown exception in regexp_match (" + imap_mailbox +
            ", '" + expr + "')";
        throw std::runtime_error(error);
    }
}

}
