#include "imap_command.h"
#include "background/per_session_event_handler.h"
#include "response/capability.hpp"
#include <common/crc.h>
#include <typed_log/typed_log.h>
#include <yplatform/encoding/base64.h>
#include <yplatform/util.h>
#include <yplatform/yield.h>
#include <boost/regex.hpp>

#define CRLF_SIZE 2

namespace yimap {

using yplatform::iequals;
typedef string::const_iterator StringConstIterator;
typedef yplatform::detail::base64_traits<StringConstIterator>::binary_t Base64DecodedIterator;

enum class AuthType
{
    plainLogin,
    plainAuthenticate,
    xoauth,
    xoauth2,
    unsupported
};

struct ContinuationData
{
    bool rejected = false;
    string buffer;

    operator bool() const
    {
        return rejected || buffer.size();
    }

    size_t size() const
    {
        return buffer.size();
    }
};

struct AuthCommand : ImapCommand
{
    using YieldCtx = yplatform::yield_context<AuthCommand>;

    AuthType authType;
    ContinuationData continuationData;
    string login;
    string secret;

    ErrorCode ec;
    Future<backend::AuthResult> authFuture;
    backend::AuthResult authResult;
    UserSettings settings;
    FolderListPtr folders;

    AuthCommand(ImapCommandArgs& cmdArgs) : ImapCommand(cmdArgs)
    {
    }

    string commandDump() const override
    {
        return tag() + " " + command() + " " + login;
    }

    void exec() override
    {
        yplatform::spawn(ioService(), yplatform::shared_from(this));
    }

    void operator()(YieldCtx yieldCtx)
    {
        reenter(yieldCtx)
        {
            if (alreadyAuthenticated()) return completeAlreadyAuthenticated();
            if (!trustedConnection()) return completeUntrustedConnection();

            authType = getAuthType();
            if (!supported(authType)) return completeBadAuthType();
            if (!validArgsFor(authType)) return completeSyntaxError();

            if (requireContinuationData())
            {
                sendContinuationRequest();
                yield asyncReadContinuationData(yieldCtx.capture(ec, continuationData));
                if (ec) return completeShutdownSession();
                if (continuationData.rejected) return completeRejected();
                std::tie(login, secret) =
                    getCredentialsFromString(authType, continuationData.buffer);
            }
            else
            {
                std::tie(login, secret) = credentialsFromArgs(authType);
            }

            authFuture = asyncCheckAuth(authType, login, secret);
            yield authFuture.add_callback(yieldCtx);
            authResult = authResultFromFuture(authFuture);
            if (authResult.tmpFail)
            {
                reportBBError();
                return completeInternalError(authResult.errorStr);
            }
            if (authResult.loginFail)
            {
                reportRejected();
                return completeInvalidCredentials(authResult.errorStr);
            }
            if (authResult.karmaFail)
            {
                return completeNoBadKarma();
            }
            storeAuthData(authResult, imapContext);

            yield settingsBackend->loadSettings().then(yieldCtx.capture(settings));
            storeSettings(settings, imapContext);
            if (!settings.imapEnabled) return completeImapDisabled("imap disabled in settings");
            // Auth requests with application passwords will be rejected by blackbox itself
            // so we check enableAuthPlain only for disabled app passwords.
            if (plainAuth(authType) && !settings.enableAuthPlain &&
                !authResult.authByApplicationPassword)
            {
                return completeImapDisabled("auth with plain password forbidden");
            }

            yield metaBackend->getFolders().then(yieldCtx.capture(folders));
            imapContext->foldersCache.setFolders(folders);
            imapContext->sessionState.setAuthenticated();

            if (settings_->loginSettings.userJournalLogEnable)
            {
                logUserJournal();
            }

            completeOk();

            if (settings_->xivaAtLogin)
            {
                notificationsBackend->subscribe(
                    std::make_shared<PerSessionEventHandler>(networkSession));
            }
        }
    }

    void operator()(YieldCtx::exception_type exception)
    {
        try
        {
            std::rethrow_exception(exception);
        }
        catch (const FolderListError& e)
        {
            completeNoFolderListError(e.what());
        }
        catch (const std::exception& e) // TODO separate backend and runtime errors
        {
            completeInternalError(e.what());
        }
    }

    bool alreadyAuthenticated()
    {
        return !imapContext->sessionState.notAuthenticated();
    }

    bool trustedConnection()
    {
        return imapContext->sessionInfo.trustedConnection;
    }

    AuthType getAuthType()
    {
        if (iequals(command(), "LOGIN"s)) return AuthType::plainLogin;

        auto typeString = carg(1);
        if (iequals(typeString, "plain")) return AuthType::plainAuthenticate;
        if (iequals(typeString, "xoauth")) return AuthType::xoauth;
        if (settings_->oauthExtension && iequals(typeString, "xoauth2")) return AuthType::xoauth2;
        return AuthType::unsupported;
    }

    bool plainAuth(AuthType authType)
    {
        return authType == AuthType::plainLogin || authType == AuthType::plainAuthenticate;
    }

    bool supported(AuthType authType)
    {
        return authType != AuthType::unsupported;
    }

    bool validArgsFor(AuthType authType)
    {
        if (authType == AuthType::plainAuthenticate && cargsSize() == 3) return false;
        return true;
    }

    bool requireContinuationData()
    {
        return iequals(command(), "AUTHENTICATE"s) && cargsSize() != 3;
    }

    std::tuple<string, string> credentialsFromArgs(AuthType authType)
    {
        switch (authType)
        {
        case AuthType::plainLogin:
        {
            auto login = quotedArg(1);
            auto password = cargsSize() > 2 ? quotedArg(2) : "";
            return { login, password };
        }
        case AuthType::plainAuthenticate:
        case AuthType::xoauth:
        case AuthType::xoauth2:
            if (cargsSize() != 3) throw std::logic_error("missing secret");
            return getCredentialsFromString(authType, carg(2));
        default:
            throw std::runtime_error("invalid auth type");
        };
    }

    std::tuple<string, string> getCredentialsFromString(AuthType authType, const string& secret)
    {
        switch (authType)
        {
        case AuthType::plainAuthenticate:
            return extractAuthPlainData(secret);
        case AuthType::xoauth:
            return { {}, secret };
        case AuthType::xoauth2:
            return extractXOauth2Token(secret);
        default:
            throw std::runtime_error("invalid auth type");
        };
    }

    template <typename Handler>
    void asyncReadContinuationData(Handler handler)
    {
        auto readOp = [this, capture_self, handler](auto continuation) mutable -> void {
            networkSession->asyncRead(
                1, [this, self, handler, continuation](auto ec) mutable -> void {
                    if (ec)
                    {
                        handler(ec, ContinuationData{});
                        return;
                    }
                    auto&& buffer = networkSession->readBuffer();
                    if (buffer.size() >= continuationDataLimit())
                    {
                        handler(boost::asio::error::invalid_argument, ContinuationData{});
                        return;
                    }
                    if (auto continuationData = handleContinuationData(buffer))
                    {
                        networkSession->consumeReadBuffer(continuationData.size() + CRLF_SIZE);
                        handler(ErrorCode{}, continuationData);
                        return;
                    }
                    continuation(continuation);
                });
        };
        readOp(readOp);
    }

    Future<backend::AuthResult> asyncCheckAuth(
        AuthType authType,
        const string& login,
        const string& secret)
    {
        switch (authType)
        {
        case AuthType::plainLogin:
            return authBackend->asyncLogin(login, secret);
        case AuthType::plainAuthenticate:
            return authBackend->asyncLogin(login, secret);
        case AuthType::xoauth:
            return authBackend->asyncLoginOAuth(secret);
        case AuthType::xoauth2:
            return authBackend->asyncLoginOAuth(secret);
        default:
            throw std::runtime_error("invalid auth type");
        };
    }

    std::tuple<string, string> extractXOauth2Token(const string& authData)
    {
        boost::iterator_range<StringConstIterator> data(authData.begin(), authData.end());
        boost::iterator_range<Base64DecodedIterator> decodedRange = yplatform::base64_decode(data);
        string decoded(decodedRange.begin(), decodedRange.end());

        // XOAUTH2 auth data is base64 encoded string like this
        // user=someuser@example.com^Aauth=Bearer vF9dft4qmTc2Nvb3RlckBhdHRhdmlzdGEuY29tCg==^A^A
        // where ^A='\001'
        // https://developers.google.com/gmail/xoauth2_protocol

        std::vector<string> parts;
        boost::split(parts, decoded, boost::is_any_of("\001"));
        if (parts.size() != 4)
        {
            throw std::domain_error("XOAUTH2 error: invalid token parts");
        }

        boost::regex userRegexp("user=(.+)$");
        boost::smatch userMatch;

        boost::regex tokenRegexp("auth=Bearer (.+)$");
        boost::smatch tokenMatch;
        if (boost::regex_match(parts[0], userMatch, userRegexp) &&
            boost::regex_match(parts[1], tokenMatch, tokenRegexp))
        {
            return { userMatch[1], tokenMatch[1] };
        }
        else
        {
            throw std::domain_error("XOAUTH2 error: invalid request format");
        }
    }

    std::tuple<string, string> extractAuthPlainData(const string& base64AuthData)
    {
        string user;
        string password;
        try
        {
            boost::iterator_range<StringConstIterator> data(
                base64AuthData.begin(), base64AuthData.end());
            boost::iterator_range<Base64DecodedIterator> decoded = yplatform::base64_decode(data);
            Base64DecodedIterator i = decoded.begin();

            bool newWord = true;
            vector<string> words;
            while (i != decoded.end())
            {
                if (newWord)
                {
                    words.push_back("");
                    newWord = false;
                }
                if (*i == '\0')
                {
                    newWord = true;
                }
                else
                {
                    words.back() += *i;
                }
                i++;
            }
            if (words.size() < 3)
            {
                throw std::domain_error("syntax error");
            }

            user = words[1];
            password = words[2];
        }
        catch (...)
        {
            throw std::domain_error("syntax error");
        }

        return { user, password };
    }

    backend::AuthResult authResultFromFuture(Future<backend::AuthResult> future)
    {
        backend::AuthResult result;
        try
        {
            result = future.get();
        }
        catch (const std::exception& e)
        {
            result.loginFail = true;
            result.tmpFail = true;
            result.errorStr = e.what();
        }
        catch (...)
        {
            result.loginFail = true;
            result.tmpFail = true;
            result.errorStr = "unknown exception";
        }
        return result;
    }

    void storeAuthData(const backend::AuthResult& authResult, ImapContextPtr context)
    {
        context->setUserData(authResult);
        context->sessionLogger.updateSessionInfo(createShortSessionInfo(*imapContext));
    }

    void storeSettings(const UserSettings& settings, ImapContextPtr context)
    {
        context->userSettings = std::make_shared<UserSettings>(settings);
        if (settings.clientLogEnabled)
        {
            context->sessionLogger.enableClientLog();
        }
    }

    size_t continuationDataLimit()
    {
        return ImapCommand::settings().serverSettings.limits.max_auth_data_size;
    }

    void completeNoFolderListError(const string& message)
    {
        logError() << "LOGIN: user='" << imapContext->userData.login
                   << "' addr=" << imapContext->sessionInfo.remoteAddress << ':'
                   << imapContext->sessionInfo.remotePort << " suid=" << imapContext->userData.suid
                   << " uid=" << imapContext->userData.uid
                   << " storage=" << imapContext->userData.storage << " failed. reason='" << message
                   << "'";

        completeNo("[UNAVAILABLE]"s, "internal error"s, "load folders error"s);
    }

    void completeOk()
    {
        logAuth("ok");
        statistic->statsLogin(ModuleContext::granted);
        auto untaggedCapability = CapabilityResponse(*imapContext, *settings_);
        sendClient() << untaggedCapability << tag() << " OK " << command() << " Completed.\r\n";
        promise->set(RET_OK);
    }

    void completeBad(
        const string& responseCode,
        const string& clientMessage,
        const string& diagnosticMessage)
    {
        logAuth("error", diagnosticMessage);
        ImapCommand::completeBad(responseCode, clientMessage);
    }

    void completeNo(
        const string& responseCode,
        const string& clientMessage,
        const string& diagnosticMessage)
    {
        logAuth("error", diagnosticMessage);
        ImapCommand::completeNo(responseCode, clientMessage);
    }

    template <typename Range>
    ContinuationData handleContinuationData(const Range& data)
    {
        auto first = data.begin();
        auto last = data.end();
        auto lineEnd = data.begin();

        bool newlineFound = findCRLF(first, last, lineEnd);
        if (!newlineFound)
        {
            return {};
        }

        string line = string(data.begin(), lineEnd);

        if ("*" == line)
        {
            return { true, {} };
        }

        return { false, line };
    }

    template <typename Iterator>
    bool findCRLF(Iterator& first, Iterator& end, Iterator& lineEnd) const
    {
        bool crFound = false;
        while (first != end)
        {
            if (crFound && *first == '\n')
            {
                lineEnd = first - 1;
                first++;
                return true;
            }
            else if (*first == '\r')
            {
                crFound = true;
            }
            first++;
        }
        return false;
    }

    void completeRejected()
    {
        completeBad("", "Command rejected.", "command rejected");
    }

    void completeAlreadyAuthenticated()
    {
        completeBad("[CLIENTBUG]", "wrong state for this command", "already authenticated");
    }

    void completeUntrustedConnection()
    {
        completeBad(
            "[PRIVACYREQUIRED]",
            "Working without SSL/TLS encryption is not allowed. Please visit "
            "https://yandex.ru/support/mail-new/mail-clients/ssl.html",
            "untrusted connection");
    }

    void completeImapDisabled(const string& diagnosticError)
    {
        reportImapDisabled();
        completeInvalidCredentials(diagnosticError);
    }

    void completeInvalidCredentials(const string& diagnosticError)
    {
        completeNo(
            "[AUTHENTICATIONFAILED]"s, "invalid credentials or IMAP is disabled"s, diagnosticError);
    }

    void completeInternalError(const string& error)
    {
        logError() << command() << " internal error: " << error;
        completeNo("[UNAVAILABLE]"s, "internal server error"s, error);
    }

    void completeNoBadKarma()
    {
        completeNo("[AUTHENTICATIONFAILED]"s, "Ommm"s, "bad karma"s);
    }

    void completeShutdownSession()
    {
        networkSession->shutdown();
        promise->set(RET_CLOSE);
    }

    void sendContinuationRequest()
    {
        sendClient() << "+ \r\n";
    }

    void completeSyntaxError()
    {
        completeBad("", "Command syntax error.", "syntax error");
    }

    void completeBadAuthType()
    {
        completeBad("", "Command syntax error.", "invalid auth type");
    }

    void reportRejected()
    {
        statistic->statsLogin(ModuleContext::rejected);
    }

    void reportBBError()
    {
        statistic->statsLogin(ModuleContext::bb_error);
    }

    void reportImapDisabled()
    {
        statistic->statsLogin(ModuleContext::disabled);
    }

    void logAuth(const string& status, const string& reason = {})
    {
        if (imapContext->userData.email.empty()) imapContext->userData.email = login;
        typed::logAuth(imapContext, status, reason);
    }

    void logUserJournal()
    {
        try
        {
            metaBackend->journalAuth();
        }
        catch (...)
        {
        }
    }

    ExtraStatFields statExtra() const override
    {
        ExtraStatFields res;
        if (imapContext->userSettings)
        {
            res = ExtraStatFields{
                std::pair{ "rename_enabled"s,
                           std::to_string(imapContext->userSettings->renameEnabled) },
                std::pair{ "localize_imap"s,
                           std::to_string(imapContext->userSettings->localizeImap) },
                std::pair{ "enable_auto_expunge"s,
                           std::to_string(imapContext->userSettings->enableAutoExpunge) },
                std::pair{ "enable_auth_plain"s,
                           std::to_string(imapContext->userSettings->enableAuthPlain) }
            };
        }
        return res;
    }
};

CommandPtr CommandLogin(ImapCommandArgs& commandArgs)
{
    return CommandPtr(new AuthCommand(commandArgs));
}

CommandPtr CommandAuthenticate(ImapCommandArgs& commandArgs)
{
    return CommandPtr(new AuthCommand(commandArgs));
}
}
