#include "message_transfer_base.h"

#include <common/settings.h>
#include <yplatform/yield.h>

namespace yimap {

struct MoveBase : MessageTransferBase
{
    using YieldCtx = yplatform::yield_context<MoveBase>;

    seq_range messageRanges;
    UidMapPtr messages;
    FolderInfo folderInfo;
    UidSequence srcUids;
    UidSequence dstUids;

    MoveBase(ImapCommandArgs& cmdArgs) : MessageTransferBase(cmdArgs)
    {
    }

    void exec() override
    {
        yplatform::spawn(ioService(), yplatform::shared_from(this));
    }

    void operator()(YieldCtx yieldCtx)
    {
        reenter(yieldCtx)
        {
            if (!settings_->moveExtension)
            {
                completeBadCommand();
                return;
            }

            selectedFolder = imapContext->sessionState.selectedFolder;
            if (selectedFolder.readOnly())
            {
                completeReadOnly();
                return;
            }

            if (selectedFolder.info().name == getDstFolderName())
            {
                yield completeOkAndOutputDiff().then(yieldCtx);
                return;
            }

            yield loadFolderInfo(getDstFolderName()).then(yieldCtx.capture(dstFolderInfo));
            messageRanges = getMessageRanges();
            yield metaBackend->loadMessages(selectedFolder, messageRanges)
                .then(yieldCtx.capture(messages));
            if (messages->empty())
            {
                return completeEmptyMessages();
            }

            if (isSpam(selectedFolder) || isSpam(dstFolderInfo))
            {
                yield reportSpam(SpamReport::ACTION_MOVE, messages).then(yieldCtx);
            }

            srcUids = messages->toUidSequence();
            yield metaBackend->moveMessages(*messages, selectedFolder.info(), dstFolderInfo)
                .then(yieldCtx.capture(dstUids));
            sendUntaggedCopyuidResponse(srcUids, dstUids);
            yield completeOkAndOutputDiff().then(yieldCtx);
        }
    }

    void operator()(YieldCtx::exception_type exception)
    {
        try
        {
            std::rethrow_exception(exception);
        }
        catch (const Utf7EncodingError&)
        {
            completeEncodingError();
        }
        catch (const NoSuchFolderError&)
        {
            completeNoSuchFolder();
        }
        catch (const std::exception& e)
        {
            completeWithException(e);
        }
        catch (...)
        {
            completeWithUnknownException();
        }
    }

    void sendUntaggedCopyuidResponse(const UidSequence& srcUids, const UidSequence& dstUids)
    {
        sendClient() << "* OK " << makeCopyuidResponse(srcUids, dstUids) << "\r\n";
    }

    void completeReadOnly()
    {
        completeNo("[CLIENTBUG]", "Can not move from read-only folder.");
    }

    void completeBadCommand()
    {
        completeBad("", "Command syntax error.");
    }

    virtual seq_range getMessageRanges() const = 0;

    virtual string getDstFolderName() const = 0;

    virtual void completeEmptyMessages() = 0;
};

struct Move : public MoveBase
{
    Move(ImapCommandArgs& cmdArgs) : MoveBase(cmdArgs)
    {
    }

    seq_range getMessageRanges() const override
    {
        auto res = selectedFolder.seqRange(false);
        parseSeqRange(argNode(1), res);
        return res;
    }

    string getDstFolderName() const override
    {
        return quotedArg(2);
    }

    void completeEmptyMessages() override
    {
        completeNo("[CLIENTBUG]", "Failed (no messages).");
    }
};

struct UidMove : public MoveBase
{
    UidMove(ImapCommandArgs& cmdArgs) : MoveBase(cmdArgs)
    {
    }

    seq_range getMessageRanges() const override
    {
        auto res = selectedFolder.seqRange(true);
        parseSeqRange(argNode(1).children[0], res);
        return res;
    }

    string getDstFolderName() const override
    {
        return quoted_string(argNode(1).children[1].value);
    }

    void completeEmptyMessages() override
    {
        completeOk("[CLIENTBUG]", "Completed (no messages).");
    }
};

CommandPtr CommandMove(ImapCommandArgs& args)
{
    return CommandPtr(new Move(args));
}
CommandPtr CommandUidMove(ImapCommandArgs& args)
{
    return CommandPtr(new UidMove(args));
}
}
