#include "imap_command.h"
#include <backend/backend.h>
#include <common/helpers/utf7imap.h>
#include <common/convert_string.h>
#include <yplatform/yield.h>

namespace yimap {

// persist data between different fetch callbacks
struct Namespace : ImapAuthenticatedCommand
{
    using YieldCtx = yplatform::yield_context<Namespace>;

    constexpr static char delim = '|';

    set<string> sharedNamespaces;

    Namespace(ImapCommandArgs& args) : ImapAuthenticatedCommand(args)
    {
    }

    void exec() override
    {
        yplatform::spawn(ioService(), yplatform::shared_from(this));
    }

    void operator()(YieldCtx yieldCtx)
    {
        reenter(yieldCtx)
        {
            yield updateFolderList().then(yieldCtx);
            sharedNamespaces = getSharedNamespaces();

            if (sharedNamespaces.empty())
            {
                sendEmpty();
            }
            else
            {
                sendNamespaces(sharedNamespaces);
            }

            yield completeOkAndOutputDiff().then(yieldCtx);
        }
    }

    void operator()(YieldCtx::exception_type exception)
    {
        try
        {
            std::rethrow_exception(exception);
        }
        catch (const std::exception& e)
        {
            completeWithException(e);
        }
    }

    set<string> getSharedNamespaces()
    {
        return imapContext->foldersCache.getFolders()->getSharedNamespaces();
    }

    void sendEmpty()
    {
        sendClient() << "* NAMESPACE ((\"\" \"" << delim << "\")) NIL NIL\r\n";
    }

    void sendNamespaces(const set<string>& sharedNamespaces)
    {
        std::ostringstream out;
        out << "* NAMESPACE ((\"\" \"" << delim << "\")) NIL (";
        for (auto&& i : sharedNamespaces)
        {
            auto utf7ImapName = folderNameToUtf7Imap(i, delim);
            out << "(" << imap_convert_string(utf7ImapName) << " \"" << delim << "\")";
        }
        out << ")\r\n";
        sendClient() << out.str();
    }
};

CommandPtr CommandNamespace(ImapCommandArgs& args)
{
    return CommandPtr(new Namespace(args));
}

}
