#include "folder_base.h"
#include <yplatform/yield.h>

namespace yimap {

struct Rename : FolderBase
{
    using YieldCtx = yplatform::yield_context<Rename>;

    string mbox;
    string destMbox;
    FolderListPtr folders;
    DBFolderId dbMbox;
    string dbDestMbox;

    Rename(ImapCommandArgs& cmdArgs) : FolderBase(cmdArgs)
    {
    }

    void exec() override
    {
        yplatform::spawn(ioService(), yplatform::shared_from(this));
    }

    void operator()(YieldCtx yieldCtx)
    {
        reenter(yieldCtx)
        {
            mbox = quotedArgAsFolderName(1);
            destMbox = quotedArgAsFolderName(2);
            if (mbox == destMbox || (isInbox(mbox) && isInbox(destMbox))) return completeOk();
            if (isInbox(mbox) || isInbox(destMbox)) return completeNoCantRenameInbox();
            if (inboxChildrenDenied() && isInboxChild(mbox))
                return completeNoCantApplyToInboxChild();
            yield updateFolderList().then(yieldCtx);

            folders = imapContext->foldersCache.getFolders();
            if (!folders->hasFolder(mbox))
            {
                return completeNoSuchFolder();
            }

            if (folders->hasFolder(destMbox))
            {
                return completeBadFolderExists();
            }

            dbMbox = folders->getDBFolderId(mbox);
            dbDestMbox = imapContext->foldersCache.getFolders()->getOriginalName(destMbox);
            yield metaBackend->renameFolder(dbMbox, dbDestMbox).then(yieldCtx);
            yield updateFolderList().then(yieldCtx);
            yield completeOkAndOutputDiff().then(yieldCtx);
        }
    }

    void operator()(YieldCtx::exception_type exception)
    {
        try
        {
            std::rethrow_exception(exception);
        }
        catch (const FolderNameTooLargeError& e)
        {
            completeBadFolderNameToolarge(e.what());
        }
        catch (const InvalidFolderNameError& e)
        {
            completeBadFolderName(e.what());
        }
        catch (const ReservedFolderNameError& e)
        {
            completeNoReservedFolderName(e.what());
        }
        catch (const NoSuchFolderError& /*e*/)
        {
            completeNoSuchFolder();
        }
        catch (const Utf7EncodingError& e)
        {
            completeBadEncodingError(mbox, e.what());
        }
        catch (const std::exception& e)
        {
            completeNoBackendError(e.what());
        }
        catch (...)
        {
            completeNoBackendError();
        }
    }

    void completeNoCantRenameInbox()
    {
        logError() << command() << " cannot rename to INBOX";
        completeNo("[CLIENTBUG]", "Cannot apply to INBOX.");
    }

    void completeNoSuchFolder()
    {
        logError() << command() << " cannot rename folder \"" << mbox << "\" to \"" << destMbox
                   << "\": no such folder.";
        completeNo(
            "[CLIENTBUG]",
            "Cannot rename folder \"" + mbox + "\" to \"" + destMbox + "\": no such folder.");
    }

    void completeBadFolderExists()
    {
        logError() << command() << " cannot rename folder \"" << mbox << "\" to \"" << destMbox
                   << "\": folder exists.";
        completeBad(
            "[CLIENTBUG]",
            "Cannot rename folder \"" + mbox + "\" to \"" + destMbox + "\": folder exists.");
    }
};

CommandPtr CommandRename(ImapCommandArgs& commandArgs)
{
    return CommandPtr(new Rename(commandArgs));
}

}
