#pragma once

#include "expunge.hpp"
#include "fetch.hpp"
#include <common/message_list_diff.h>

namespace yimap {

// General status update response
struct StatusUpdateResponse
{
    std::vector<ExpungeResponse> expunged;
    std::vector<FetchResponse> changed;
    boost::optional<std::size_t> exists;
    boost::optional<std::size_t> recent;

    static StatusUpdateResponse create(MailboxDiffPtr diff);

    template <typename Stream>
    Stream& send(Stream& stream) const;
};

inline StatusUpdateResponse StatusUpdateResponse::create(MailboxDiffPtr diff)
{
    StatusUpdateResponse result;
    if (diff)
    {
        if (diff->hasAdded()) result.exists = diff->messageCount();

        if (diff->recentCount() > 0 && diff->recentChanged()) result.recent = diff->recentCount();

        MessageRevisionedSet deleted(diff->deleted.begin(), diff->deleted.end());
        auto offsetForModseq = 0u;
        uint64_t prevModseq = 0;
        for (const auto& message : deleted)
        {
            offsetForModseq = (message.modseq == prevModseq ? offsetForModseq + 1 : 0);
            prevModseq = message.modseq;
            result.expunged.emplace_back(
                ExpungeResponse{ message.modseq, message.uid, message.num, offsetForModseq });
        }

        for (const auto& message : diff->changed)
        {
            // We should not output fetch for added messages.
            if (!message.added)
            {
                result.changed.emplace_back(
                    FetchResponse{ message.modseq, message.uid, message.num, message.flags });
            }
        }
    }
    return result;
}

template <typename Stream>
Stream& StatusUpdateResponse::send(Stream& stream) const
{
    for (auto& expunge : expunged)
    {
        stream << expunge;
    }

    if (exists)
    {
        stream << "* " << *exists << " EXISTS\r\n";
    }

    for (auto& fetch : changed)
    {
        stream << fetch;
    }

    if (recent)
    {
        stream << "* " << *recent << " RECENT\r\n";
    }
    return stream;
}

template <typename Stream>
Stream&& operator<<(Stream&& stream, const StatusUpdateResponse& response)
{
    response.send(stream);
    return std::forward<Stream>(stream);
}

} // namespace yimap
