#include "factory.h"
#include "tree_parsers.h"
#include <common/lex_ids.h>
#include <common/tree.h>
#include <common/types.h>

namespace yimap {

SearchNodePtr createSearchNode(
    const TreeNode& treeNode,
    const SearchNodes& children,
    const string& charset);

SearchNodePtr buildSearchTree(const TreeNode& treeNode, const string& charset, size_t maxHeight)
{
    if (maxHeight == 0) throw std::domain_error("search tree height limit exceeded");

    SearchNodes children;
    for (auto&& child : treeNode.children)
    {
        children.push_back(buildSearchTree(child, charset, maxHeight - 1));
    }
    return createSearchNode(treeNode, children, charset);
}

bool allNodesAreExpressions(const SearchNodes& children);
std::vector<string> collectSearchExpression(const SearchNodes& children);

SearchNodePtr createSearchNode(
    const TreeNode& treeNode,
    const SearchNodes& children,
    const string& charset)
{
    long scopelexeme = getScopeLexeme(treeNode);

    if (children.size() && allNodesAreExpressions(children))
    {
        auto expressions = collectSearchExpression(children);
        return std::make_shared<SearchByExpression>(makeLogicExpression(scopelexeme, expressions));
    }

    switch (scopelexeme)
    {
    case lex_ids::SEARCH_KEY_BODY:
    case lex_ids::SEARCH_KEY_TEXT:
    case lex_ids::SEARCH_KEY_SUBJECT:
    case lex_ids::SEARCH_KEY_FROM:
    case lex_ids::SEARCH_KEY_TO:
    case lex_ids::SEARCH_KEY_CC:
    case lex_ids::SEARCH_KEY_BCC:
        return std::make_shared<SearchByExpression>(makeFieldExpression(treeNode, charset));

    case lex_ids::SEARCH_KEY_HEADER:
        if (yplatform::iequals(firstChildText(treeNode), "message-id"))
            return std::make_shared<SearchNodeMessageId>(getMessageID(treeNode));
        else
            return std::make_shared<SearchByExpression>(makeHeaderExpression(treeNode, charset));

    case lex_ids::SEARCH_KEY_SENTBEFORE:
    case lex_ids::SEARCH_KEY_SENTON:
    case lex_ids::SEARCH_KEY_SENTSINCE:
        return std::make_shared<SearchNodeDoNothing>();

    case lex_ids::SEARCH_KEY_OR:
        return std::make_shared<SearchNodeOr>(children);

    case lex_ids::SEARCH_KEY_NOT:
        return std::make_shared<SearchNodeNot>(children);

    case lex_ids::SEARCH_KEY:
    case lex_ids::SEARCH_KEY_AND:
        return std::make_shared<SearchNodeAnd>(children);

    case lex_ids::SEARCH_KEY_ALL:
        return std::make_shared<SearchNodeDoNothing>();

    case lex_ids::SEARCH_KEY_SEQSET:
        return std::make_shared<SearchNodeSeqset>(treeNode);
    case lex_ids::SEARCH_KEY_UID:
        return std::make_shared<SearchNodeSeqsetUid>(treeNode);

    case lex_ids::SEARCH_KEY_ANSWERED:
    case lex_ids::SEARCH_KEY_DELETED:
    case lex_ids::SEARCH_KEY_FLAGGED:
    case lex_ids::SEARCH_KEY_NEW:
    case lex_ids::SEARCH_KEY_OLD:
    case lex_ids::SEARCH_KEY_RECENT:
    case lex_ids::SEARCH_KEY_SEEN:
    case lex_ids::SEARCH_KEY_UNANSWERED:
    case lex_ids::SEARCH_KEY_UNDELETED:
    case lex_ids::SEARCH_KEY_UNFLAGGED:
    case lex_ids::SEARCH_KEY_UNSEEN:
    case lex_ids::SEARCH_KEY_DRAFT:
    case lex_ids::SEARCH_KEY_UNDRAFT:
    case lex_ids::SEARCH_KEY_KEYWORD:
    case lex_ids::SEARCH_KEY_UNKEYWORD:
    {
        auto [set, unset] = getSetUnsetFlags(treeNode);
        return std::make_shared<SearchNodeFlags>(set, unset);
    }

    case lex_ids::SEARCH_KEY_BEFORE:
    case lex_ids::SEARCH_KEY_ON:
    case lex_ids::SEARCH_KEY_SINCE:
    {
        auto [min, max] = getDateRange(treeNode);
        return std::make_shared<SearchNodeDate>(min, max);
    }

    case lex_ids::SEARCH_KEY_LARGER:
    case lex_ids::SEARCH_KEY_SMALLER:
    {
        auto [min, max] = getSizeRange(treeNode);
        return SearchNodePtr(new SearchNodeSize(min, max));
    }

    default:
        return std::make_shared<SearchNodeDoNothing>();
    }
}

bool allNodesAreExpressions(const SearchNodes& children)
{
    for (auto&& child : children)
    {
        if (nodeCast<SearchByExpression>(child) == nullptr) return false;
    }
    return true;
}

std::vector<string> collectSearchExpression(const SearchNodes& children)
{
    assert(allNodesAreExpressions(children));
    std::vector<string> expressions;
    for (auto&& child : children)
    {
        if (auto casted = nodeCast<SearchByExpression>(child))
        {
            expressions.push_back(casted->searchExpression);
        }
    }
    return expressions;
}
}
