#pragma once

#include <common/tree.h>
#include <common/folder.h>
#include <common/uid_map.h>
#include <common/sequence_ranges.h>
#include <common/types.h>
#include <memory>

namespace yimap {

struct SearchNode;
typedef std::shared_ptr<SearchNode> SearchNodePtr;
typedef std::vector<SearchNodePtr> SearchNodes;

struct SearchNode
{
    SearchNodes children;

    SearchNode(const SearchNodes& children = {}) : children(children)
    {
    }

    virtual ~SearchNode() = default;

    virtual string nodeName() const = 0;
};

struct SearchNodeStatic : SearchNode
{
    virtual UidMapPtr reduce(UidMapPtr bounds) = 0;
};

struct SearchNodeDoNothing : SearchNodeStatic
{
    UidMapPtr reduce(UidMapPtr bounds) override
    {
        return bounds;
    }

    string nodeName() const override
    {
        return "SearchNodeDoNothing"s;
    }
};

struct SearchNodeLogical : SearchNode
{
    SearchNodeLogical(const SearchNodes& children) : SearchNode(children)
    {
    }

    virtual UidMapPtr reduce(UidMapPtr bounds, const std::vector<UidMapPtr>& childrenResults) = 0;
};

struct SearchNodeAnd : SearchNodeLogical
{
    SearchNodeAnd(const SearchNodes& children) : SearchNodeLogical(children)
    {
    }

    UidMapPtr reduce(UidMapPtr bounds, const std::vector<UidMapPtr>& childrenResults) override
    {
        auto result = clone(bounds);
        for (auto&& childResult : childrenResults)
        {
            result = result->intersect(childResult);
        }
        return result;
    }

    string nodeName() const override
    {
        return "SearchNodeAnd"s;
    }
};

struct SearchNodeOr : SearchNodeLogical
{
    SearchNodeOr(const SearchNodes& children) : SearchNodeLogical(children)
    {
    }

    UidMapPtr reduce(UidMapPtr bounds, const std::vector<UidMapPtr>& childrenResults) override
    {
        auto result = std::make_shared<UidMap>();
        for (auto&& childResult : childrenResults)
        {
            result = result->with(childResult);
        }
        return result->intersect(bounds);
    }

    string nodeName() const override
    {
        return "SearchNodeOr"s;
    }
};

struct SearchNodeNot : SearchNodeLogical
{
    SearchNodeNot(const SearchNodes& children) : SearchNodeLogical(children)
    {
    }

    UidMapPtr reduce(UidMapPtr bounds, const std::vector<UidMapPtr>& childrenResults) override
    {
        auto result = clone(bounds);
        for (auto&& childResult : childrenResults)
        {
            result = result->without(childResult);
        }
        return result;
    }

    string nodeName() const override
    {
        return "SearchNodeNot"s;
    }
};

struct SearchNodeSeqset : SearchNode
{
    const TreeNode& tree;
    bool uidMode = false;

    SearchNodeSeqset(const TreeNode& tree) : tree(tree)
    {
        validateSeqRange(tree);
    }

    seq_range seqRangeFor(FolderRef& mailbox)
    {
        auto seq = mailbox.seqRange(uidMode);
        parseSeqRangeDeprecated(tree, seq);
        return seq;
    }

    UidMapPtr reduce(UidMapPtr bounds, FolderRef& mailbox)
    {
        auto result = clone(bounds);
        result = result->filterBySequence(seqRangeFor(mailbox));
        return result;
    }

    string nodeName() const override
    {
        return "SearchNodeSeqset"s;
    }
};

struct SearchNodeSeqsetUid : SearchNodeSeqset
{
    SearchNodeSeqsetUid(const TreeNode& tree) : SearchNodeSeqset(tree)
    {
        uidMode = true;
    }

    string nodeName() const override
    {
        return "SearchNodeSeqsetUid"s;
    }
};

struct SearchNodeFlags : SearchNodeStatic
{
    std::set<string> set, unset;

    SearchNodeFlags(const std::set<string>& set, const std::set<string>& unset)
        : set(set), unset(unset)
    {
    }

    UidMapPtr reduce(UidMapPtr bounds) override
    {
        auto result = clone(bounds);
        return result->filterByFlags(set, unset);
    }

    string nodeName() const override
    {
        return "SearchNodeFlags"s;
    }
};

struct SearchNodeDate : SearchNodeStatic
{
    time_t min, max;

    SearchNodeDate(time_t min, time_t max) : min(min), max(max)
    {
    }

    UidMapPtr reduce(UidMapPtr bounds) override
    {
        auto matchTime = [this](const MessageData& msg) {
            return min <= msg.time && msg.time <= max;
        };
        auto result = clone(bounds);
        return result->filterMessages(matchTime);
    }

    string nodeName() const override
    {
        return "SearchNodeDate"s;
    }
};

struct SearchNodeSize : SearchNodeStatic
{
    size_t min, max;

    SearchNodeSize(size_t min, size_t max) : min(min), max(max)
    {
    }

    UidMapPtr reduce(UidMapPtr bounds) override
    {
        auto matchSize = [this](const MessageData& msg) {
            assert(msg.details);
            return min <= msg.details.size && msg.details.size <= max;
        };
        auto result = clone(bounds);
        return result->filterMessages(matchSize);
    }

    string nodeName() const override
    {
        return "SearchNodeSize"s;
    }
};

struct SearchNodeMessageId : SearchNode
{
    string messageId;
    // XXX results are stored in SearchNodeMessageId objects
    // to simplify asynchronous command code.
    UidMapPtr found;

    SearchNodeMessageId(const string& messageId) : messageId(messageId)
    {
    }

    UidMapPtr reduce(UidMapPtr bounds)
    {
        if (!found) throw std::runtime_error("incorrect search node state");
        auto foundUidSet = found->toUidSet();
        auto filter = [&](auto&& msg) { return foundUidSet->find(msg.uid) != foundUidSet->end(); };
        auto result = clone(bounds);
        return result->filterMessages(filter);
    }

    string nodeName() const override
    {
        return "SearchNodeMessageId"s;
    }
};

struct SearchByExpression : SearchNode
{
    string searchExpression;
    // XXX results are stored in SearchByExpression objects
    // to simplify asynchronous command code.
    std::set<string> found;

    SearchByExpression(const string& expression) : searchExpression(expression)
    {
    }

    UidMapPtr reduce(UidMapPtr bounds)
    {
        auto result = clone(bounds);
        auto filter = [&](auto&& msg) { return found.count(msg.smid()); };
        return result->filterMessages(filter);
    }

    string nodeName() const override
    {
        return "SearchByExpression"s;
    }
};

template <typename T, typename Y>
inline auto nodeCast(const std::shared_ptr<Y>& node)
{
    return std::dynamic_pointer_cast<T>(node);
}
}
