#include <common/tree.h>
#include <common/errors.h>
#include <common/types.h>
#include <yplatform/util/date_parse.h>
#include <yplatform/encoding/url_encode.h>
#include <yplatform/convert/yconv.h>

#include <boost/regex.hpp>
#include <boost/algorithm/string.hpp>
#include <boost/algorithm/string/trim_all.hpp>

namespace yimap {

inline long getScopeLexeme(const TreeNode& treeNode)
{
    return treeNode.value.id().to_long();
}

inline string getValueString(const TreeNode& keyNode)
{
    return string(keyNode.value.begin(), keyNode.value.end());
}

inline string firstChildText(const TreeNode& node)
{
    if (node.children.size() == 0) return "";
    return string(node.children[0].value.begin(), node.children[0].value.end());
}

inline string composeExpression(const string& condition, const std::vector<string>& expressions)
{
    assert(expressions.size());
    std::ostringstream request_os;
    request_os << "(" << expressions[0];
    for (size_t i = 1; i < expressions.size(); i++)
    {
        request_os << " " << condition << " " << expressions[i];
    }
    request_os << ")";
    return request_os.str();
}

inline string makeLogicExpression(long scope_lexeme, const std::vector<string>& expressions)
{
    assert(expressions.size());

    switch (scope_lexeme)
    {
    case lex_ids::SEARCH_KEY_NOT:
        if (expressions.size() != 1) throw InvalidFormat("too many expressions");
        return "(NOT "s + expressions[0] + ")"s;

    case lex_ids::SEARCH_KEY_OR:
        return composeExpression("OR"s, expressions);

    case lex_ids::SEARCH_KEY:
    case lex_ids::SEARCH_KEY_AND:
        return composeExpression("AND"s, expressions);
    default:
        throw std::runtime_error("unknown logic node");
    };
}

inline std::pair<std::set<string>, std::set<string>> getSetUnsetFlags(const TreeNode& node)
{
    std::set<string> set;
    std::set<string> unset;
    switch (getScopeLexeme(node))
    {
    case lex_ids::SEARCH_KEY_ANSWERED:
        set.insert("\\Answered");
        break;
    case lex_ids::SEARCH_KEY_DELETED:
        set.insert("\\Deleted");
        break;
    case lex_ids::SEARCH_KEY_DRAFT:
        set.insert("\\Draft");
        break;
    case lex_ids::SEARCH_KEY_FLAGGED:
        set.insert("\\Flagged");
        break;
    case lex_ids::SEARCH_KEY_NEW:
        set.insert("\\Recent");
        unset.insert("\\Seen");
        break;
    case lex_ids::SEARCH_KEY_OLD:
        unset.insert("\\Recent");
        break;
    case lex_ids::SEARCH_KEY_RECENT:
        set.insert("\\Recent");
        break;
    case lex_ids::SEARCH_KEY_SEEN:
        set.insert("\\Seen");
        break;
    case lex_ids::SEARCH_KEY_UNANSWERED:
        unset.insert("\\Answered");
        break;
    case lex_ids::SEARCH_KEY_UNDELETED:
        unset.insert("\\Deleted");
        break;
    case lex_ids::SEARCH_KEY_UNDRAFT:
        unset.insert("\\Draft");
        break;
    case lex_ids::SEARCH_KEY_UNFLAGGED:
        unset.insert("\\Flagged");
        break;
    case lex_ids::SEARCH_KEY_UNSEEN:
        unset.insert("\\Seen");
        break;
    case lex_ids::SEARCH_KEY_UNKEYWORD:
        unset.insert(getValueString(node));
        break;
    case lex_ids::SEARCH_KEY_KEYWORD:
        set.insert(getValueString(node));
        break;
    }

    return { set, unset };
}

inline const boost::regex esc("["
                              "\\["
                              "\\]"
                              "\\{"
                              "\\}"
                              "\\!"
                              "\\+"
                              "\\-"
                              "\\*"
                              "\\^"
                              "\\?"
                              "\\/"
                              "'"
                              "~"
                              "\""
                              "("
                              ")"
                              ":"
                              " "
                              "]");

inline const boost::regex fixQuote("\\\\"
                                   "\\\\"
                                   "\"");

inline string escapeRequest(const string& requestString)
{
    const std::string rep("\\\\$0");
    string result =
        boost::regex_replace(requestString, esc, rep, boost::match_default | boost::format_perl);
    result = boost::regex_replace(
        result,
        fixQuote,
        "\\\\"
        "\\\"",
        boost::match_default | boost::format_perl);
    boost::algorithm::trim_fill_if(result, " ", boost::algorithm::is_cntrl());
    return result;
}

inline auto getDateRange(const TreeNode& treeNode)
{
    yplatform::util::ddmmyyyy date;
    if (!yplatform::util::parse_ddmmyyyy(date, treeNode.value.begin(), treeNode.value.end()))
    {
        throw InvalidFormat("invalid date format");
    }

    // FIXME if really needed: code below does not count time zones, neither leap seconds
    struct tm tm1, tm2;
    ::memset(&tm1, 0, sizeof tm1);
    ::memset(&tm2, 0, sizeof tm2);

    tm1.tm_year = (date.yyyy - 1900);
    tm1.tm_mon = date.mm;
    tm1.tm_mday = date.dd;
    tm1.tm_hour = 0;
    tm1.tm_min = 0;
    tm1.tm_sec = 0;
    time_t t1 = ::timegm(&tm1);

    tm2.tm_year = (date.yyyy - 1900);
    tm2.tm_mon = date.mm;
    tm2.tm_mday = date.dd;
    tm2.tm_hour = 23;
    tm2.tm_min = 59;
    tm2.tm_sec = 59;
    time_t t2 = ::timegm(&tm2);

    if (t1 == static_cast<time_t>(-1) || t2 == static_cast<time_t>(-1))
    {
        throw InvalidFormat("invalid date format");
    }

    switch (getScopeLexeme(treeNode))
    {
    case lex_ids::SEARCH_KEY_SINCE:
        t2 = static_cast<time_t>(std::numeric_limits<uint32_t>::max());
        break;
    case lex_ids::SEARCH_KEY_BEFORE:
        t2 = t1;
        t1 = static_cast<time_t>(0);
        break;
    }

    return std::pair{ t1, t2 };
}

inline auto getSizeRange(const TreeNode& treeNode)
{
    uint32_t value;
    try
    {
        value = boost::lexical_cast<uint32_t>(getValueString(treeNode));
    }
    catch (...)
    {
        throw InvalidFormat("invalid message size value");
    }

    uint32_t min = 0;
    uint32_t max = std::numeric_limits<uint32_t>::max();

    if (getScopeLexeme(treeNode) == lex_ids::SEARCH_KEY_LARGER)
    {
        min = value + 1;
    }
    else
    {
        max = value;
    }

    return std::pair{ min, max };
}

inline auto getMessageID(const TreeNode& treeNode)
{
    return string(treeNode.children[1].value.begin(), treeNode.children[1].value.end());
}

string makeHeaderExpression(const string& headerName, const string& searchText);

inline string makeFieldExpression(const TreeNode& treeNode, const string& charset)
{
    static const map<long, string> lexemeToScope = {
        { lex_ids::SEARCH_KEY_BODY, "body" },       { lex_ids::SEARCH_KEY_BCC, "bcc" },
        { lex_ids::SEARCH_KEY_CC, "cc" },           { lex_ids::SEARCH_KEY_FROM, "from" },
        { lex_ids::SEARCH_KEY_SUBJECT, "subject" }, { lex_ids::SEARCH_KEY_TO, "to" },
        { lex_ids::SEARCH_KEY_TEXT, "text" }
    };

    auto field = getValueString(treeNode);
    try
    {
        field = yplatform::convert::iconvert_string("utf-8", charset, field);
    }
    catch (const std::runtime_error&)
    {
        throw UnsupportedEncodingError();
    }
    field = escapeRequest(field);
    string searchScope;
    auto found = lexemeToScope.find(getScopeLexeme(treeNode));
    if (found != lexemeToScope.end()) searchScope = found->second;
    return makeHeaderExpression(searchScope, field);
}

inline string makeHeaderExpression(const TreeNode& treeNode, const string& charset)
{
    if (treeNode.children.size() < 2)
    {
        throw InvalidFormat("invalid header format");
    }
    string name(treeNode.children[0].value.begin(), treeNode.children[0].value.end());
    string value(treeNode.children[1].value.begin(), treeNode.children[1].value.end());
    try
    {
        value = yplatform::convert::iconvert_string("utf-8", charset, value);
    }
    catch (const std::runtime_error&)
    {
        throw UnsupportedEncodingError();
    }
    value = escapeRequest(value);
    return makeHeaderExpression(boost::algorithm::to_lower_copy(name), value);
}

inline string makeHeaderExpression(const string& headerName, const string& searchText)
{
    std::stringstream ss;
    if (headerName == "to")
    {
        ss << "(hdr_to:(*" << searchText << "*) OR hdr_to_keyword:(*" << searchText << "*))";
    }
    else if (headerName == "cc")
    {
        ss << "(hdr_cc:(*" << searchText << "*) OR hdr_cc_keyword:(*" << searchText << "*))";
    }
    else if (headerName == "bcc")
    {
        ss << "(hdr_bcc:(*" << searchText << "*) OR hdr_bcc_keyword:(*" << searchText << "*))";
    }
    else if (headerName == "from")
    {
        ss << "(hdr_from:(*" << searchText << "*) OR hdr_from_keyword:(*" << searchText << "*))";
    }
    else if (headerName == "subject")
    {
        ss << "(hdr_subject:(*" << searchText << "*) OR hdr_subject_keyword:(*" << searchText
           << "*))";
    }
    else if (headerName == "body")
    {
        ss << "(body_text:(" << searchText << "*) OR pure_body:(" << searchText
           << "*)"
              ")";
    }
    else if (headerName == "text")
    {
        ss << "("
              "body_text:"
              "("
           << searchText
           << "*)"
              " OR "
              "pure_body:"
              "("
           << searchText
           << "*)"
              " OR "
              "hdr_bcc:"
              "(*"
           << searchText
           << "*)"
              " OR "
              "hdr_bcc_keyword:"
              "(*"
           << searchText
           << "*)"
              " OR "
              "hdr_cc:"
              "(*"
           << searchText
           << "*)"
              " OR "
              "hdr_cc_keyword:"
              "(*"
           << searchText
           << "*)"
              " OR "
              "hdr_from:"
              "(*"
           << searchText
           << "*)"
              " OR "
              "hdr_from_keyword:"
              "(*"
           << searchText
           << "*)"
              " OR "
              "hdr_subject:"
              "(*"
           << searchText
           << "*)"
              " OR "
              "hdr_subject_keyword:(*"
           << searchText
           << "*)"
              " OR "
              "hdr_to:"
              "(*"
           << searchText
           << "*)"
              " OR "
              "hdr_to_keyword:"
              "(*"
           << searchText
           << "*)"
              ")";
    }
    else
    {
        ss << "(headers:(" << headerName << "\\:\\ *" << searchText << "*))";
    }
    return ss.str();
}

}