#include "imap_command.h"
#include <common/convert_string.h>
#include <common/settings.h>
#include <boost/algorithm/string/join.hpp>
#include <yplatform/yield.h>

namespace yimap {

struct Status : ImapAuthenticatedCommand
{
    using YieldCtx = yplatform::yield_context<Status>;

    std::optional<FolderInfo> cachedFolder;
    FolderInfo folder;

    Status(ImapCommandArgs& cmdArgs) : ImapAuthenticatedCommand(cmdArgs)
    {
    }

    struct RequiredInfo
    {
        bool messagesCount = false;
        bool recentCount = false;
        bool unseenCount = false;
        bool uidNext = false;
        bool uidValidity = false;
    };

    void exec() override
    {
        yplatform::spawn(ioService(), yplatform::shared_from(this));
    }

    void operator()(YieldCtx yieldCtx)
    {
        reenter(yieldCtx)
        {
            std::tie(displayName, requiredInfo) = extractArgsFromAST();

            if (!checkEncoding(displayName))
            {
                completeBadEncoding(displayName);
                return;
            }

            if (useCache()) cachedFolder = getFolderFromCache(displayName);
            if (cachedFolder)
            {
                sendFolderInfo(*cachedFolder);
                yield completeOkAndOutputDiff().then(yieldCtx);
                return;
            }
            yield loadFolderInfo(displayName).then(yieldCtx.capture(folder));
            if (useCache()) storeFolderToCache(displayName, folder);
            sendFolderInfo(folder);
            yield completeOkAndOutputDiff().then(yieldCtx);
        }
    }

    void operator()(YieldCtx::exception_type exception)
    {
        try
        {
            std::rethrow_exception(exception);
        }
        catch (const NoSuchFolderError&)
        {
            completeNoSuchFolder(displayName);
        }
        catch (const std::exception& e)
        {
            completeNoBackendError(displayName, e.what());
        }
    }

    std::tuple<string, RequiredInfo> extractArgsFromAST()
    {
        string displayName = quotedArg(1);
        RequiredInfo info;
        auto& children = tree->data.trees[0].children;
        for (auto& node : children[2].children)
        {
            switch (static_cast<lex_ids::ids>(node.value.id().to_long()))
            {
            case lex_ids::STATUS_ATT_MESSAGES:
                info.messagesCount = true;
                break;

            case lex_ids::STATUS_ATT_RECENT:
                info.recentCount = true;
                break;

            case lex_ids::STATUS_ATT_UNSEEN:
                info.unseenCount = true;
                break;

            case lex_ids::STATUS_ATT_UIDNEXT:
                info.uidNext = true;
                break;

            case lex_ids::STATUS_ATT_UIDVALIDITY:
                info.uidValidity = true;
                break;

            default:
                logError() << "bad field requested: " << node.value.id().to_long();
                assert(false); // this should not happen
                break;
            }
        }
        return std::make_tuple(std::move(displayName), std::move(info));
    }

    bool useCache()
    {
        // For some clients use additional cache with longer ttl.
        size_t cacheTTL = settings_->getStatusCacheTtl(imapContext->sessionInfo.remoteAddress);
        return cacheTTL > 0 && imapContext->statusCache;
    }

    std::optional<FolderInfo> getFolderFromCache(const string& displayName)
    {
        return imapContext->statusCache->getFolder(displayName);
    }

    void storeFolderToCache(const string& displayName, const FolderInfo& folder)
    {
        size_t ttl = settings_->getStatusCacheTtl(imapContext->sessionInfo.remoteAddress);
        imapContext->statusCache->setFolder(displayName, folder, ttl);
    }

    void sendFolderInfo(const FolderInfo& folder)
    {
        std::vector<string> info;
        if (requiredInfo.messagesCount)
        {
            info.emplace_back("MESSAGES " + std::to_string(folder.messageCount));
        }
        if (requiredInfo.recentCount)
        {
            info.emplace_back("RECENT " + std::to_string(folder.recentCount));
        }
        if (requiredInfo.unseenCount)
        {
            info.emplace_back("UNSEEN " + std::to_string(folder.unseenCount));
        }
        if (requiredInfo.uidNext)
        {
            info.emplace_back("UIDNEXT " + std::to_string(folder.uidNext));
        }
        if (requiredInfo.uidValidity)
        {
            info.emplace_back("UIDVALIDITY " + std::to_string(folder.uidValidity));
        }
        sendClient() << "* STATUS " << imap_convert_string(displayName) << " ("
                     << boost::algorithm::join(info, " ") << ")\r\n";
    }

    void completeBadEncoding(const string& folder)
    {
        logError() << "status error: bad encoding, folder=" << folder;
        completeBad("[CLIENTBUG]", "Folder encoding error.");
    }

    void completeNoSuchFolder(const string& folder)
    {
        logError() << "status error: no such folder, folder=" << folder;
        completeNo("[TRYCREATE]", "No such folder.");
    }

    void completeNoBackendError(const string& folder, const string& message)
    {
        logError() << "status error: " << message << ", folder=" << folder;
        completeNo("[UNAVAILABLE]", "Backend error");
    }

    string displayName;
    RequiredInfo requiredInfo;
};

CommandPtr CommandStatus(ImapCommandArgs& commandArgs)
{
    return CommandPtr(new Status(commandArgs));
}

}
