#include <backend/misc/spam_report.h>

#include "imap_command.h"
#include <common/sequence_ranges.h>
#include <common/settings.h>
#include <boost/algorithm/string.hpp>
#include <yplatform/yield.h>

namespace yimap {

struct StoreBase : ImapSelectedCommand
{
    using YieldCtx = yplatform::yield_context<StoreBase>;

    FolderRef selectedFolder;
    seq_range messageRanges;
    UidMapPtr messages;

    Flags flagsToAdd;
    Flags flagsToDelete;

    std::string spamFlag;
    const bool uidMode;

    MailboxDiffPtr diff;
    bool expunged = false;

    StoreBase(ImapCommandArgs& cmdArgs, bool uidMode)
        : ImapSelectedCommand(cmdArgs), uidMode(uidMode)
    {
    }

    void exec() override
    {
        yplatform::spawn(ioService(), yplatform::shared_from(this));
    }

    void operator()(YieldCtx yieldCtx)
    {
        reenter(yieldCtx)
        {
            selectedFolder = imapContext->sessionState.selectedFolder;
            if (selectedFolder.readOnly())
            {
                completeReadOnly();
                return;
            }

            if (uidMode)
            {
                yield updateFolderAndSendDiff().then(yieldCtx);
            }

            messageRanges = getMessageRanges();
            yield metaBackend->loadMessages(selectedFolder, messageRanges)
                .then(yieldCtx.capture(messages));
            if (messages->empty())
            {
                return completeEmptyMessages();
            }

            extractFlagsFromArgs();
            if (spamFlag.size())
            {
                yield reportSpam().then(yieldCtx);
            }

            if (flagsToAdd.hasFlag("\\Deleted") && autoExpungeEnabled())
            {
                expunged = true;
                yield metaBackend->expunge(selectedFolder, messages).then(yieldCtx);
                completeOkWithFakeDeletedFlag();
            }
            else
            {
                yield metaBackend->updateFlags(selectedFolder, messages, flagsToDelete, flagsToAdd)
                    .then(yieldCtx.capture(diff));
                completeOkWithFlags(diff);
            }
        }
    }

    void operator()(YieldCtx::exception_type exception)
    {
        try
        {
            std::rethrow_exception(exception);
        }
        catch (const std::exception& e)
        {
            completeWithException(e);
        }
        catch (...)
        {
            completeWithUnknownException();
        }
    }

    void extractFlagsFromArgs()
    {
        auto flags = getFlagsValues();
        spamFlag = findSpamFlag(flags);
        fillAddAndDeleteFlags(flags);
        flagsToDelete.delFlag(Flags::MSG_RECENT);
        if (selectedFolder.isShared())
        {
            flagsToAdd.delFlag(Flags::MSG_DELETED);
        }
    }

    Future<void> reportSpam()
    {
        return metaBackend->loadBodyMetadata(*messages->toSmidList())
            .then([=, capture_self](auto future) {
                auto bodyMeta = future.get();
                auto spamReport = SpamReport::make(
                    *imapContext,
                    *messages,
                    bodyMeta,
                    SpamReport::ACTION_FLAG,
                    selectedFolder.info().name,
                    "",
                    spamFlag);
                spamReport->logToTskv();
            });
    }

    string findSpamFlag(const Flags& flags)
    {
        for (auto flag : settings_->reportSpam.spamFlags)
        {
            if (flags.hasFlag(flag))
            {
                return flag;
            }
        }

        for (auto flag : settings_->reportSpam.notSpamFlags)
        {
            if (flags.hasFlag(flag))
            {
                return flag;
            }
        }
        return ""s;
    }

    void fillAddAndDeleteFlags(const Flags& flags)
    {
        auto actionLexId = getActionLexId();
        switch (actionLexId)
        {
        case lex_ids::STORE_ATT_SET_FLAGS:
            fillAddAndDeleteFlagsForSetOp(flags);
            break;

        case lex_ids::STORE_ATT_ADD_FLAGS:
            flagsToAdd = flags;
            break;

        case lex_ids::STORE_ATT_DEL_FLAGS:
            flagsToDelete = flags;
            break;

        default:
            assert(false && "unknown action in store command");
        }
    }

    void fillAddAndDeleteFlagsForSetOp(const Flags& flags)
    {
        flagsToAdd = flags;
        for (auto& msg : *messages)
        {
            flagsToDelete.addFlags(msg.flags);
        }
        flagsToDelete.delFlags(flagsToAdd);
    }

    Flags parseFlagsNode(TreeNode flagsNode) const
    {
        Flags flags;
        for (TreeNode const& node : flagsNode.children)
        {
            string flag(node.value.begin(), node.value.end());
            if (flag.empty()) continue; // should not normally happen - XXX log ?

            if (yplatform::iequals(flag, "\\Seen")) flags.setFlag(Flags::MSG_SEEN);
            if (yplatform::iequals(flag, "\\Answered")) flags.setFlag(Flags::MSG_ANSWERED);
            if (yplatform::iequals(flag, "\\Flagged")) flags.setFlag(Flags::MSG_FLAGGED);
            if (yplatform::iequals(flag, "\\Deleted")) flags.setFlag(Flags::MSG_DELETED);
            if (yplatform::iequals(flag, "\\Draft")) flags.setFlag(Flags::MSG_DRAFT);

            if (!flag.empty() && flag[0] != '\\')
            {
                flags.setFlag(flag);
            }
        }
        return flags;
    }

    bool autoExpungeEnabled()
    {
        return imapContext->userSettings->enableAutoExpunge &&
            imapContext->settings->serverSettings.enableAutoExpunge;
    }

    void completeOkWithFlags(MailboxDiffPtr diff)
    {
        if (verbose())
        {
            for (const auto& msg : diff->changed)
            {
                sendClient() << "* " << msg.num << " FETCH (UID " << msg.uid << " FLAGS ("
                             << static_cast<string>(msg.flags) << "))\r\n";
            }
        }
        completeOk();
    }

    void completeOkWithFakeDeletedFlag()
    {
        if (verbose())
        {
            for (auto&& msg : *messages)
            {
                sendClient() << "* " << msg.num << " FETCH (UID " << msg.uid
                             << " FLAGS (\\Deleted))\r\n";
            }
        }
        completeOk();
    }

    void completeReadOnly()
    {
        completeBad("[CLIENTBUG]", "Can not store in read-only folder.");
    }

    ExtraStatFields statExtra() const override
    {
        return ExtraStatFields{
            std::pair{ "expunged"s, std::to_string(expunged) },
        };
    }

    virtual void completeEmptyMessages() = 0;

    virtual seq_range getMessageRanges() const = 0;

    virtual lex_ids::ids getActionLexId() const = 0;

    virtual bool verbose() const = 0;

    virtual Flags getFlagsValues() const = 0;
};

class Store : public StoreBase
{
public:
    Store(ImapCommandArgs& cmdArgs) : StoreBase(cmdArgs, false)
    {
    }

    seq_range getMessageRanges() const override
    {
        auto res = selectedFolder.seqRange(false);
        parseSeqRange(argNode(1), res);
        return res;
    }

    lex_ids::ids getActionLexId() const override
    {
        return lex_ids::ids(argNode(2).value.id().to_long());
    }

    bool verbose() const override
    {
        return argNode(3).value.id().to_long() == lex_ids::STORE_ATT_VERBOSE;
    }

    Flags getFlagsValues() const override
    {
        auto flagsNode = argNode(4);
        return parseFlagsNode(flagsNode);
    }

    void completeEmptyMessages() override
    {
        completeNo("[CLIENTBUG]", "Failed (no messages).");
    }
};

class UidStore : public StoreBase
{
public:
    UidStore(ImapCommandArgs& cmdArgs) : StoreBase(cmdArgs, true)
    {
    }

    seq_range getMessageRanges() const override
    {
        auto res = selectedFolder.seqRange(true);
        parseSeqRange(argNode(1).children[0], res);
        return res;
    }

    lex_ids::ids getActionLexId() const override
    {
        return lex_ids::ids(argNode(1).children[1].value.id().to_long());
    }

    bool verbose() const override
    {
        return argNode(1).children[2].value.id().to_long() == lex_ids::STORE_ATT_VERBOSE;
    }

    Flags getFlagsValues() const override
    {
        auto flagsNode = argNode(1).children[3];
        return parseFlagsNode(flagsNode);
    }

    void completeEmptyMessages() override
    {
        completeOk("[CLIENTBUG]", "Completed (no messages).");
    }
};

CommandPtr CommandStore(ImapCommandArgs& commandArgs)
{
    return CommandPtr(new Store(commandArgs));
}

CommandPtr CommandUidStore(ImapCommandArgs& commandArgs)
{
    return CommandPtr(new UidStore(commandArgs));
}

}
