#include "imap_command.h"
#include <common/quoted_string.h>

#include <backend/backend.h>
#include <common/helpers/utf7imap.h>
#include <yplatform/yield.h>

namespace yimap {

struct ChangeSubscribedStateBase : ImapAuthenticatedCommand
{
    using YieldCtx = yplatform::yield_context<ChangeSubscribedStateBase>;

    string folderName;
    bool targetStateSubscribed = true;

    ChangeSubscribedStateBase(ImapCommandArgs& cmdArgs) : ImapAuthenticatedCommand(cmdArgs)
    {
    }

    void exec() override
    {
        yplatform::spawn(ioService(), yplatform::shared_from(this));
    }

    void operator()(YieldCtx yieldCtx)
    {
        reenter(yieldCtx)
        {
            folderName = quotedArg(1);
            checkEncodingThrows(folderName);
            yield updateFolderList().then(yieldCtx);
            yield changeSubscribedState(folderName, targetStateSubscribed).then(yieldCtx);
            yield completeOkAndOutputDiff().then(yieldCtx);
        }
    }

    void operator()(YieldCtx::exception_type exception)
    {
        try
        {
            std::rethrow_exception(exception);
        }
        catch (const InvalidFolderNameError& e)
        {
            completeInvalidFolderName(e.what());
        }
        catch (const Utf7EncodingError& e)
        {
            completeInvalidEncoding(e.what());
        }
        catch (const ReservedFolderNameError& e)
        {
            completeNoSuchFolder(e.what());
        }
        catch (const std::exception& e)
        {
            completeBackendError(e.what());
        }
        catch (...)
        {
            completeBackendError();
        }
    }

    Future<void> changeSubscribedState(const string& folderName, bool targetStateSubscribed)
    {
        return metaBackend->subscribe(folderIdFromName(folderName), targetStateSubscribed);
    }

    DBFolderId folderIdFromName(const string& folderName)
    {
        DBFolderId id;
        if (imapContext->foldersCache.hasFolder(folderName))
        {
            id = imapContext->foldersCache.getFolders()->getDBFolderId(folderName);
        }
        else
        {
            id.name = folderName;
        }
        return id;
    }

    void completeInvalidEncoding(const string& error)
    {
        logDebug() << command() << " cannot convert from utf7: " << error;
        completeBad("[CLIENTBUG]", "Folder encoding error.");
    }

    void completeNoSuchFolder(const string& error)
    {
        logDebug() << command() << " reserved folder name: " << error;
        completeNo("[UNAVAILABLE]", "No such folder.");
    }

    void completeInvalidFolderName(const string& error)
    {
        logDebug() << command() << " Bad folder name: " << error;
        completeBad("[CLIENTBUG]", "Bad folder name.");
    }

    void completeBackendError(const string& error = {})
    {
        logError() << command() << error;
        completeNo("[UNAVAILABLE]", "Backend error.");
    }
};

struct Subscribe : ChangeSubscribedStateBase
{
    Subscribe(ImapCommandArgs& cmdArgs) : ChangeSubscribedStateBase(cmdArgs)
    {
        targetStateSubscribed = true;
    }
};

struct Unsubscribe : ChangeSubscribedStateBase
{
    Unsubscribe(ImapCommandArgs& cmdArgs) : ChangeSubscribedStateBase(cmdArgs)
    {
        targetStateSubscribed = false;
    }
};

CommandPtr CommandSubscribe(ImapCommandArgs& args)
{
    return CommandPtr(new Subscribe(args));
}
CommandPtr CommandUnsubscribe(ImapCommandArgs& args)
{
    return CommandPtr(new Unsubscribe(args));
}
}
