#pragma once

#include <common/types.h>

#include <cctype>
#include <cstddef>

#include <yplatform/util.h>

#include <boost/unordered_set.hpp>
#include <boost/range.hpp>
#include <boost/function.hpp>

namespace yimap {

namespace detail {

inline bool icmp(char c1, char c2)
{
    return std::tolower(c1) < std::tolower(c2);
}

struct icompare
{
    bool operator()(const string& x, const string& y) const
    {
        return std::lexicographical_compare(x.begin(), x.end(), y.begin(), y.end(), icmp);
    }
};

} // namespace detail

class Flags
{
public:
    Flags() : stdFlags_(0)
    {
    }

    Flags(const std::initializer_list<string>& flags) : stdFlags_(0)
    {
        addFlags(flags);
    }

    // standard system flags
    enum MessageFlag
    {
        _MSG_ERROR_ = 0,
        MSG_SEEN,
        MSG_ANSWERED,
        MSG_FLAGGED,
        MSG_DELETED,
        MSG_DRAFT,
        MSG_RECENT,

        // This is extra flag, not from RFC
        MSG_FORWARDED
    };

public:
    bool empty() const
    {
        return stdFlags_ == 0 && userFlags_.empty();
    }

    void setFlag(MessageFlag const& flag);
    template <typename RangeT>
    void setFlag(RangeT const& rng);

    void delFlag(MessageFlag const& flag);
    template <typename RangeT>
    void delFlag(RangeT const& rng);

    bool hasFlag(MessageFlag const& flag) const;
    bool hasFlag(const string& rng) const;

    void delFlags(Flags const& flags);

    template <typename SetT>
    void delFlags(SetT const& flags);

    void addFlags(Flags const& flags);

    template <typename SetT>
    void addFlags(SetT const& flags);

    template <typename SetT>
    bool hasAllFlags(SetT const& flags) const;

    template <typename SetT>
    bool hasAtLeastOneFlag(SetT const& flags) const;

    operator string() const;
    void forEachFlag(std::function<void(string const&)> hook) const;

    inline bool operator==(Flags const& other) const
    {
        return (stdFlags_ == other.stdFlags_ && userFlags_ == other.userFlags_);
    }

    inline bool operator!=(Flags const& other) const
    {
        return !this->operator==(other);
    }

protected:
    template <typename RangeT>
    MessageFlag parseFlag(RangeT const& rng) const;

private:
    int stdFlags_;
    std::set<string, detail::icompare> userFlags_;
};

template <typename RangeT>
Flags::MessageFlag Flags::parseFlag(RangeT const& rng) const
{
    if (::yplatform::iequals(rng, "\\Seen")) return MSG_SEEN;
    if (::yplatform::iequals(rng, "\\Answered")) return MSG_ANSWERED;
    if (::yplatform::iequals(rng, "\\Flagged")) return MSG_FLAGGED;
    if (::yplatform::iequals(rng, "\\Deleted")) return MSG_DELETED;
    if (::yplatform::iequals(rng, "\\Draft")) return MSG_DRAFT;
    if (::yplatform::iequals(rng, "\\Recent")) return MSG_RECENT;
    if (::yplatform::iequals(rng, "$Forwarded")) return MSG_FORWARDED;
    return _MSG_ERROR_;
}

template <typename RangeT>
void Flags::setFlag(RangeT const& rng)
{
    MessageFlag f = parseFlag(rng);
    if (f != _MSG_ERROR_) stdFlags_ = stdFlags_ | (1 << f);
    else
        userFlags_.insert(boost::copy_range<string>(rng));
}

template <typename RangeT>
void Flags::delFlag(RangeT const& rng)
{
    MessageFlag f = parseFlag(rng);
    if (f != _MSG_ERROR_) stdFlags_ &= ~(1 << f);
    else
        userFlags_.erase(boost::copy_range<string>(rng));
}

inline bool Flags::hasFlag(const string& flag) const
{
    bool rc;
    MessageFlag f = parseFlag(flag);
    if (f != _MSG_ERROR_) rc = stdFlags_ & (1 << f);
    else
        rc = (userFlags_.find(flag) != userFlags_.end());
    return rc;
}

inline bool Flags::hasFlag(MessageFlag const& f) const
{
    return stdFlags_ & (1 << f);
}

template <typename SetT>
void Flags::addFlags(SetT const& flags)
{
    for (typename SetT::const_iterator i = flags.begin(); i != flags.end(); ++i)
    {
        setFlag(*i);
    }
}

template <typename SetT>
void Flags::delFlags(SetT const& flags)
{
    for (typename SetT::const_iterator i = flags.begin(); i != flags.end(); ++i)
    {
        delFlag(*i);
    }
}

template <typename SetT>
bool Flags::hasAllFlags(SetT const& flags) const
{
    for (typename SetT::const_iterator i = flags.begin(); i != flags.end(); ++i)
    {
        if (!hasFlag(*i)) return false;
    }
    return true;
}

template <typename SetT>
bool Flags::hasAtLeastOneFlag(SetT const& flags) const
{
    for (typename SetT::const_iterator i = flags.begin(); i != flags.end(); ++i)
    {
        if (hasFlag(*i)) return true;
    }
    return false;
}

}
