#pragma once

#include <common/types.h>
#include <common/folder_info.h>
#include <common/message_data.h>
#include <common/message_list_diff.h>
#include <common/seq_range.h>
#include <common/log/logger.h>

#include <yplatform/log.h>

#include <boost/thread/mutex.hpp>
#include <boost/thread/locks.hpp>
#include <boost/shared_ptr.hpp>

#include <mutex>
#include <vector>

namespace yimap {

typedef boost::mutex mutex_t;
typedef boost::unique_lock<mutex_t> lock_t;

class Folder
{
public:
    FolderInfo folderInfo;
    // Folder owner suid.
    string fsuid;
    mutex_t mux;
    const bool isSharedFolder;

public:
    Folder(const FolderInfo& aFolderInfo, bool isShared = false)
        : folderInfo(aFolderInfo), isSharedFolder(isShared)
    {
    }
    virtual ~Folder()
    {
    }

    uint32_t revision() const
    {
        return folderInfo.revision;
    }
    uint32_t messages_count() const
    {
        return folderInfo.messageCount;
    }
    uint32_t recent_messages() const
    {
        return folderInfo.recentCount;
    }
    uint32_t max_uid()
    {
        return folderInfo.uidNext > 0 ? folderInfo.uidNext - 1 : 0;
    }
    string name() const
    {
        return folderInfo.name;
    }
    string fid() const
    {
        return folderInfo.fid;
    }

    FolderInfo copyFolderInfo() const
    {
        return folderInfo;
    }

    virtual bool isShared()
    {
        return isSharedFolder;
    }
    virtual size_t knownRevision() const
    {
        return folderInfo.revision;
    }
    virtual bool hasChanges() const
    {
        return false;
    }

    // Mailbox modification.
    virtual void update(MessagesVector& messages) = 0;
    virtual void exchange(MessagesVector& /*messages*/, FolderInfoPtr /*newFolderInfo*/)
    {
    }
    virtual MailboxDiffPtr updateToRevision(bool onlyChanged) = 0;

    virtual void insertMessages(MessagesVector& messages) = 0;

    virtual void filterPartialUids(const UidVector& uids, UidVector& result) = 0;

    // Selecting messages range
    virtual UidMapPtr filterByRanges(const seq_range& ranges, MessageData::Predicate pred)
        const = 0;
    virtual UidMap filterContinuouslyCachedByRanges(const seq_range& ranges, size_t limit)
        const = 0;
    virtual std::tuple<size_t, seq_range> uncachedRanges(const seq_range& ranges) const = 0;

    virtual void checkConsistency(Logger&)
    {
    }
    virtual void debugDump(Logger&)
    {
    }
    virtual string dump() const
    {
        return {};
    }

private:
    Folder(const Folder& other);
};

typedef std::shared_ptr<Folder> FolderPtr;

//-----------------------------------------------------------------------------

typedef std::unique_lock<std::mutex> std_lock;

class FolderRef
{
public:
    FolderRef() : readOnly_(false)
    {
    }

    FolderRef(FolderPtr aFolder, bool readOnly) : folder(aFolder), readOnly_(readOnly)
    {
        if (!aFolder) throw std::runtime_error("invalid folder ptr");
    }

    FolderRef(const FolderRef& that) = default;
    FolderRef(FolderRef&& that) = default;
    FolderRef& operator=(const FolderRef& that) = default;
    FolderRef& operator=(FolderRef&& that) = default;

    string fid() const
    {
        return folder->fid();
    }

    size_t knownRevision() const
    {
        return folder->knownRevision();
    }

    operator bool() const
    {
        return folder.get();
    }

    bool readOnly()
    {
        return readOnly_;
    }

    bool empty()
    {
        return folder->messages_count() == 0;
    }

    FolderInfo info() const
    {
        return folder->copyFolderInfo();
    }

    seq_range seqRange(bool uidMode) const
    {
        return seq_range(1, uidMode ? folder->max_uid() : folder->messages_count(), uidMode);
    }

    seq_range fullUidRange() const
    {
        seq_range fullRange(1, folder->max_uid(), true);
        fullRange.insert(range_t(1, folder->max_uid()));
        return fullRange;
    }

    void insertMessages(MessagesVector& messages)
    {
        folder->insertMessages(messages);
    }

    void filterPartialUids(const UidVector& uids, UidVector& result)
    {
        folder->filterPartialUids(uids, result);
    }

    std::tuple<size_t, seq_range> uncachedRanges(const seq_range& range) const
    {
        return folder->uncachedRanges(range);
    }

    UidMapPtr filterByRanges(const seq_range& range, MessageData::Predicate pred = MessageData::id)
        const
    {
        return folder->filterByRanges(range, pred);
    }

    UidMap filterContinuouslyCachedByRanges(const seq_range& range, size_t limit) const
    {
        return folder->filterContinuouslyCachedByRanges(range, limit);
    }

    // Note: 'messages' array could be changed to improve performance
    void update(MessagesVector& messages)
    {
        if (messages.empty()) return;
        folder->update(messages);
    }

    void exchange(MessagesVector& messages, FolderInfoPtr newFolderInfo = FolderInfoPtr())
    {
        folder->exchange(messages, newFolderInfo);
    }

    MailboxDiffPtr updateToRevision(bool onlyChanged = false)
    {
        return folder->updateToRevision(onlyChanged);
    }

    bool hasChanges() const
    {
        return folder->hasChanges();
    }

    void checkConsistency(Logger& logger)
    {
        return folder->checkConsistency(logger);
    }

    void debugDump(Logger& logger)
    {
        return folder->debugDump(logger);
    }

    bool isShared() const
    {
        return folder->isShared();
    }

    string dump() const
    {
        return folder->dump();
    }

    void dump(yplatform::log::source& logger, std::string&& where, MessagesVector& messages)
    {
        std::stringstream logMess;

        logMess << "DUMP LOG: in <" << where << "> method"
                << "\r\n";

        for (auto& mess : messages)
        {
            logMess << "DUMP LOG: "
                    << "num:" << std::to_string(mess.num) << " "
                    << "uid:" << std::to_string(mess.uid) << " "
                    << "mid:" << std::to_string(mess.mid) << " "
                    << "modseq:" << std::to_string(mess.modseq) << " "
                    << "recent:" << std::to_string(mess.recent) << " "
                    << "added:" << std::to_string(mess.added) << " "
                    << "deleted:" << std::to_string(mess.deleted) << " "
                    << "baseUid:" << std::to_string(mess.baseUid) << " "
                    << "offset:" << std::to_string(mess.offset) << " "
                    << "\r\n";
        }

        YLOG(logger, info) << logMess.str();
    }

private:
    FolderPtr folder;
    bool readOnly_;
};

typedef map<uint32_t, uint32_t> UintMap;
typedef std::map<uint32_t, MessageData> UidMessagesSet;
typedef UidMessagesSet::const_iterator ConstUidIterator;
typedef boost::iterator_range<ConstUidIterator> UidRange;

struct UidMapEntry
{
    uint32_t uid;
    // Number of messages, from previous base message to this.
    uint32_t chain;
    uint32_t nextBase;
};

typedef std::map<uint32_t, UidMapEntry> UidMapData;
typedef boost::shared_ptr<UidMapData> UidMapDataPtr;

}
