#include <common/folder_list.h>
#include <common/helpers/utf7imap.h>

#include <common/settings.h>
#include <common/lang_config.h>
#include <yplatform/util.h>
#include <boost/algorithm/string.hpp>
#include <boost/lexical_cast.hpp>

namespace yimap {

bool FolderList::FListCmp::operator()(const string& a, const string& b) const
{
    const char delimiter = '|';
    string::const_iterator ia = a.begin();
    string::const_iterator ib = b.begin();
    while (ia != a.end() && ib != b.end())
    {
        if (*ia == delimiter && *ib == delimiter)
        {
            // do nothing
        }
        else if (*ia == delimiter)
        {
            return true;
        }
        else if (*ib == delimiter)
        {
            return false;
        }
        else if (*ia != *ib)
        {
            return *ia < *ib;
        }
        ia++;
        ib++;
    }
    if (ia == a.end() && ib == b.end()) return false;
    return ia == a.end();
}

bool FolderList::FListCmp::operator()(const FullFolderInfoPtr& a, const FullFolderInfoPtr& b) const
{
    return (*this)(a->name, b->name);
}

FolderList::FolderList(RawFolderListPtr rawList, LanguageConfigPtr languageSettings)
    : languageSettings(languageSettings)
{
    _inboxListed = languageSettings->nameFromSymbol("inbox");
    std::sort(rawList->begin(), rawList->end(), FListCmp());

    RawFolderList originalFolders;
    initRenamedFolders(rawList, originalFolders);

    initUsualFolders(originalFolders);

    generateMissedHierarchy();
    findChildren();
}

void FolderList::initRenamedFolders(RawFolderListPtr list, RawFolderList& originalFolders)
{
    auto it = list->cbegin();
    while (it != list->cend())
    {
        FullFolderInfoPtr srcFolder = (*it);
        bool hasSymbol = !srcFolder->symbol.empty();
        if (hasSymbol)
        {
            auto secondName = languageSettings->nameFromSymbol(srcFolder->symbol);
            if (secondName.empty())
            {
                originalFolders.push_back(srcFolder);
            }
            else
            {
                string utf7Name = folderNameToUtf7Imap(secondName, '|');
                _listedToDBNames[utf7Name] = srcFolder;

                it = remapChildren(it, list->end(), secondName);
                continue;
            }
        }
        else
        {
            originalFolders.push_back(srcFolder);
        }

        it++;
    }
}

void FolderList::initUsualFolders(const RawFolderList& list, bool force)
{
    RawFolderList duplicates;
    auto it = list.begin();
    while (it != list.end())
    {
        FullFolderInfoPtr srcFolder = (*it);
        string utf7Name = folderNameToUtf7Imap(srcFolder->name, '|');
        if (_listedToDBNames.find(utf7Name) == end())
        {
            _listedToDBNames[utf7Name] = srcFolder;
        }
        else
        {
            // it changes inside, to process all child of current folder
            if (!force)
            {
                duplicates.push_back(srcFolder);
                string name = srcFolder->name + "|";
                it++;
                while (it != list.end() && boost::starts_with((*it)->name, name))
                {
                    duplicates.push_back(*it);
                    it++;
                }
            }
            else
            {
                for (int h = 0;; h++)
                {
                    string postfix = "_" + boost::lexical_cast<string>(h);

                    string newAmazingName = unifyInbox(srcFolder->name + postfix);
                    newAmazingName = folderNameToUtf7Imap(newAmazingName, '|');

                    if (_listedToDBNames.find(newAmazingName) == _listedToDBNames.end())
                    {
                        _listedToDBNames[newAmazingName] = srcFolder;
                        it = remapChildren(it, list.end(), newAmazingName);
                        break;
                    }
                }
            }
            // to not additionally change it
            continue;
        }
        it++;
    }

    if (!duplicates.empty())
    {
        initUsualFolders(duplicates, true);
    }
}

RawFolderList::const_iterator FolderList::remapChildren(
    RawFolderList::const_iterator it,
    RawFolderList::const_iterator end,
    const string& newName)
{
    string newParentName = newName + "|";
    string oldParentName = (*it)->name + "|";
    it++;

    while (it != end && boost::starts_with((*it)->name, oldParentName))
    {
        string newChildName = (*it)->name;
        boost::replace_first(newChildName, oldParentName, newParentName);
        _listedToDBNames[folderNameToUtf7Imap(newChildName, '|')] = *it;
        it++;
    }
    return it;
}

string FolderList::xlistFromSymbol(const string& folderSymbol) const
{
    return languageSettings->xlistFromSymbol(folderSymbol);
}

bool FolderList::checkInbox(const std::string& str) const
{
    string folderName = str;
    boost::algorithm::to_lower(folderName);
    return folderName == "inbox";
}

void FolderList::generateMissedHierarchy()
{
    Container fakeFolders;
    for (auto& currentFolder : _listedToDBNames)
    {
        string name = currentFolder.first;
        while (true)
        {
            size_t pos = name.rfind('|');
            if (pos == string::npos) break;

            string parent = name.substr(0, pos);
            if (_listedToDBNames.find(parent) != _listedToDBNames.end()) break;

            FullFolderInfoPtr ptr(new FullFolderInfo());
            ptr->name = parent;
            ptr->symbol = "zombie";
            fakeFolders[parent] = ptr;
            name = parent;
        }
    }

    _listedToDBNames.insert(fakeFolders.begin(), fakeFolders.end());
}

void FolderList::findChildren()
{
    Container::iterator end = _listedToDBNames.end();
    for (Container::iterator curr = _listedToDBNames.begin(); curr != end; ++curr)
    {
        Container::iterator next = curr;
        next++;
        if (next == end) break;

        const string& currName = curr->first;
        const string& nextName = next->first;

        // We should check that "currName|" is prefix of nextName
        string prefix = currName + "|";
        pair<string::const_iterator, string::const_iterator> res =
            std::mismatch(prefix.begin(), prefix.end(), nextName.begin());
        if (res.first == prefix.end())
        {
            curr->second->hasChildren = true;
        }
    }
}

string FolderList::symbolByFid(const string& fid) const
{
    for (const value_type& i : _listedToDBNames)
    {
        if (i.second->fid == fid) return i.second->symbol;
    }
    return "";
}

bool FolderList::isTrash(const string& fid) const
{
    return yplatform::iequals("trash", symbolByFid(fid));
}

bool FolderList::isSpam(const string& fid) const
{
    return yplatform::iequals("spam", symbolByFid(fid));
}

bool FolderList::isInbox(const string& fid) const
{
    return yplatform::iequals("inbox", symbolByFid(fid));
}

bool FolderList::isFolderSelectable(const string& folderName) const
{
    ensureValidFolderName(folderName);
    Container::const_iterator found = _listedToDBNames.find(unifyInbox(folderName));
    if (found != _listedToDBNames.end()) return found->second->symbol != "zombie";

    return true;
}

string FolderList::getOriginalName(const string& listedName) const
{
    string name = unifyInbox(listedName);
    Container::const_iterator found = _listedToDBNames.find(name);
    if (found != _listedToDBNames.end()) return found->second->name;
    else
    {
        string decodedName = getValidUtf8FolderName(name);
        for (auto const& entry : _listedToDBNames)
        {
            if (entry.second->name == decodedName)
            {
                throw ReservedFolderNameError(listedName);
            }
        }
    }

    return getValidUtf8FolderName(name);
}

string FolderList::unifyInbox(const string& folderName) const
{
    size_t pos = folderName.find_first_of('|');

    if (checkInbox(folderName.substr(0, pos)))
    {
        if (pos == string::npos)
        {
            return _inboxListed;
        }
        else
        {
            string temp = folderName;
            temp.replace(0, pos, _inboxListed);
            return temp;
        }
    }

    return folderName;
}

bool FolderList::hasFolder(const string& folder) const
{
    ensureValidFolderName(folder);
    string folderName = unifyInbox(folder);
    return _listedToDBNames.find(folderName) != _listedToDBNames.end();
}

DBFolderId FolderList::getDBFolderId(const string& folderName) const
{
    ensureValidFolderName(folderName);
    auto it = _listedToDBNames.find(unifyInbox(folderName));
    if (it == _listedToDBNames.end())
    {
        throw NoSuchFolderError(folderName, "folder not found in cache");
    }

    auto folderPtr = it->second;
    DBFolderId res;
    res.fid = folderPtr->fid;
    res.name = folderPtr->name;

    return res;
}

bool FolderList::getSharedStatusByFid(const string& fid) const
{
    for (const value_type& folderEntry : _listedToDBNames)
    {
        if (fid == folderEntry.second->fid)
        {
            return folderEntry.second->isShared;
        }
    }
    return false;
}

set<string> FolderList::getSharedNamespaces() const
{
    set<string> result;
    for (const Container::value_type& folder : _listedToDBNames)
    {
        if (folder.second->isShared)
        {
            const string& folderName = folder.first;
            string::size_type delPos = folderName.find("|");
            if (delPos != string::npos)
            {
                string foundNamespace = folderName.substr(0, delPos + 1); // +1 - include '|'
                result.insert(foundNamespace);
            }
        }
    }
    return result;
}

void FolderList::ensureValidFolderName(const string& listedName) const
{
    if (listedName.empty())
    {
        throw InvalidFolderNameError("empty folder name");
    }
    if (listedName.front() == '|' || listedName.back() == '|')
    {
        throw InvalidFolderNameError("empty hierarchy element");
    }
    if (listedName.find("||") != string::npos)
    {
        throw InvalidFolderNameError("empty hierarchy element");
    }
    folderNameFromUtf7Imap(listedName, '|'); // check encoding
}

string FolderList::getValidUtf8FolderName(const string& listedName) const
{
    string name;
    string prefix;
    splitName(unifyInbox(listedName), prefix, name);

    if (name.empty()) throw InvalidFolderNameError("empty hierarchy element");
    name = folderNameFromUtf7Imap(name, '|');

    if (prefix.empty()) return name;
    else
        return getOriginalName(prefix) + "|" + name;
}

void FolderList::splitName(const string& from, string& prefix, string& name) const
{
    size_t last = from.find_last_of('|');

    if (last == string::npos)
    {
        prefix = "";
        name = from;
    }
    else
    {
        prefix = from.substr(0, last);
        name = from.substr(last + 1);
    }
}

}
