#pragma once

#include <common/types.h>
#include <common/folder_info.h>

#include <deque>
#include <set>
#include <map>
#include <stdexcept>

namespace yimap {

struct DBFolderId
{
    DBFolderId()
    {
    }
    DBFolderId(const string& name, const string& fid) : name(name), fid(fid)
    {
    }
    string name;
    string fid;

    bool operator<(const DBFolderId& other) const
    {
        return std::tie(name, fid) < std::tie(other.name, other.fid);
    }
};

using namespace std;

class LanguageConfig;
using LanguageConfigPtr = std::shared_ptr<const LanguageConfig>;

typedef std::deque<FullFolderInfoPtr> RawFolderList;
typedef std::shared_ptr<RawFolderList> RawFolderListPtr;

class FolderList
{
    // We should sort presuming, that delimiter '|' should be lower than any other symbol
    struct FListCmp
    {
        bool operator()(const string& a, const string& b) const;
        bool operator()(const FullFolderInfoPtr& a, const FullFolderInfoPtr& b) const;
    };

public:
    explicit FolderList()
    {
    }
    explicit FolderList(RawFolderListPtr rawList, LanguageConfigPtr languageSettings);
    ~FolderList()
    {
    }

    // string nameBySymbol(const string& symbol) const;
    string xlistFromSymbol(const string& folderSymbol) const;

    string symbolByFid(const string& fid) const;
    bool isTrash(const string& fid) const;
    bool isSpam(const string& fid) const;
    bool isInbox(const string& fid) const;

    string getOriginalName(const string& listedName) const;
    bool hasFolder(const string& folderName) const;
    bool isFolderSelectable(const string& folderName) const;
    DBFolderId getDBFolderId(const string& folderName) const;

    bool getSharedStatusByFid(const string& fid) const;
    set<string> getSharedNamespaces() const;

    typedef map<string, FullFolderInfoPtr, FListCmp> Container;
    typedef Container::iterator iterator;
    typedef Container::const_iterator const_iterator;
    typedef Container::value_type value_type;

    bool empty() const
    {
        return _listedToDBNames.empty();
    }
    size_t size() const
    {
        return _listedToDBNames.size();
    }
    iterator begin()
    {
        return _listedToDBNames.begin();
    }
    iterator end()
    {
        return _listedToDBNames.end();
    }
    const_iterator begin() const
    {
        return _listedToDBNames.begin();
    }
    const_iterator end() const
    {
        return _listedToDBNames.end();
    }
    FullFolderInfoPtr at(string key) const
    {
        return _listedToDBNames.at(unifyInbox(key));
    }

    Container getInnerMap() const
    {
        return _listedToDBNames;
    }

private:
    void initRenamedFolders(RawFolderListPtr list, RawFolderList& lastFolders);
    void initUsualFolders(const RawFolderList& list, bool force = false);

    RawFolderList::const_iterator remapChildren(
        RawFolderList::const_iterator it,
        RawFolderList::const_iterator end,
        const string& newName);

    string unifyInbox(const string& folderName) const;
    bool checkInbox(const string& str) const;

    void ensureValidFolderName(const string& listedName) const;
    string getValidUtf8FolderName(const string& listedName) const;
    void splitName(const string& from, string& prefix, string& name) const;

    void generateMissedHierarchy();
    void findChildren();

    std::string _inboxListed;
    LanguageConfigPtr languageSettings;
    Container _listedToDBNames;
};

typedef std::shared_ptr<FolderList> FolderListPtr;

//-----------------------------------------------------------------------------
// Folder list errors

struct InvalidFolderNameError : public runtime_error
{
    explicit InvalidFolderNameError(const string& reason)
        : runtime_error(string("Invalid folder name :") + reason)
    {
    }
};

struct ReservedFolderNameError : public runtime_error
{
    explicit ReservedFolderNameError(const string& name)
        : runtime_error(string("Reserved folder name :") + name)
    {
    }
};

struct NoSuchFolderError : public runtime_error
{
    explicit NoSuchFolderError(const string& folderName, const string& message)
        : runtime_error(string("No such folder '") + folderName + "': " + message)
    {
    }
};

}
