#pragma once

#include "../module_context.h"

#include <common/messages_stats.h>
#include <common/types.h>
#include <common/settings.h>
#include <common/session_info.h>
#include <common/user_data.h>
#include <common/folder.h>
#include <common/folder_list.h>
#include <common/seq_range.h>
#include <common/status_cache.h>
#include <common/log/session_logger.h>
#include <common/stats.h>

#include <boost/thread.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/unordered_map.hpp>
#include <boost/noncopyable.hpp>
#include <boost/date_time/posix_time/posix_time_types.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/range/iterator_range.hpp>
#include <boost/thread/shared_mutex.hpp>

namespace yimap {

typedef boost::function<void()> void_hook_t;
typedef std::function<void(void_hook_t)> StrandHook;

// User specific localization parameters
class LanguageConfig;

class ImapContext
    : public Context
    , private boost::noncopyable
{
public:
    ImapContext(IOService& io) : ioService(io)
    {
    }

    virtual const string& get_name() const
    {
        static const string NAME = "imap_context";
        return NAME;
    }

    SessionLogger sessionLogger{ ShortSessionInfo(uniq_id()), {}, {} };
    const boost::posix_time::ptime initTime = boost::posix_time::second_clock::local_time();
    auto age() const
    {
        return boost::posix_time::second_clock::local_time() - initTime;
    }

    void setUserData(const UserData& data)
    {
        userData = data;
        auto time = Clock::now();
        auto dur = time.time_since_epoch();
        auto seconds = dur.count() * Clock::period::num / Clock::period::den;
        userData.authTime = seconds;
    }

    auto getModuleContext()
    {
        auto moduleContext = create_module_data<ModuleContext>("imap_server");
        moduleContext->setModuleStat(stats);
        return moduleContext;
    }

    void calcStats(const std::string& commandName)
    {
        getModuleContext()->stats(1, 0, boost::algorithm::to_upper_copy(commandName));
    }

    UserData userData;
    SessionInfo sessionInfo;
    SettingsCPtr settings;
    UserSettingsCPtr userSettings;
    StatsPtr stats;
    MessagesStats messagesStats;

    enum class SessionState
    {
        INIT,
        AUTH,
        SELECT
    };

    struct
    {
        SessionState state = SessionState::INIT;

        bool wasLogout = false;
        FolderRef selectedFolder;

        string state_name() const
        {
            switch (state)
            {
            case SessionState::AUTH:
                return "AUTH";
            case SessionState::INIT:
                return "INIT";
            default:
                return "SELECT";
            }
        }
        void setAuthenticated()
        {
            state = SessionState::AUTH;
        }
        bool notAuthenticated() const
        {
            return state == SessionState::INIT;
        }
        bool selected() const
        {
            return state == SessionState::SELECT;
        }
        bool checkAppropriateState(SessionState requiredState)
        {
            switch (requiredState)
            {
            case SessionState::INIT:
                return true;
            case SessionState::AUTH:
                return state != SessionState::INIT;
            case SessionState::SELECT:
                return state == SessionState::SELECT;
            }
        }
        void selectFolder(FolderPtr folder)
        {
            state = SessionState::SELECT;
            selectedFolder = FolderRef(folder, false);
        }
        void selectFolderReadOnly(FolderPtr folder)
        {
            state = SessionState::SELECT;
            selectedFolder = FolderRef(folder, true);
        }

        void unselect()
        {
            state = SessionState::AUTH;
            selectedFolder = FolderRef();
        }
    } sessionState;

    struct SearchState
    {
        TimePoint penultRequestTs = Clock::now();
        TimePoint lastRequestTs = Clock::now();
        std::size_t requestsCount = 0;

        void updateStats()
        {
            ++requestsCount;
            penultRequestTs = lastRequestTs;
            lastRequestTs = Clock::now();
        }
    } searchState;

    StatusCachePtr statusCache{ new StatusCache() }; // status command cache

    struct
    {
        FolderListPtr getFolders()
        {
            return folders;
        }
        void setFolders(FolderListPtr newFolders)
        {
            folders = newFolders;
        }
        bool hasFolder(const string& name) const
        {
            return folders && folders->hasFolder(name);
        }
        bool getSharedStatusByFid(const string& fid) const
        {
            return folders->getSharedStatusByFid(fid);
        }

    private:
        FolderListPtr folders;
    } foldersCache;

    struct
    {
        std::atomic_size_t badCommands = { 0 }; // TODO move to stats
    } processingLoopState;

    IOService& ioService;

protected:
    virtual boost::shared_ptr<boost::property_tree::ptree> core_get_stat() const;
};

typedef boost::shared_ptr<ImapContext> ImapContextPtr;
typedef boost::shared_ptr<const ImapContext> ImapContextConstPtr;
typedef boost::shared_ptr<const ImapContext> ImapContextConstPtr;
typedef boost::weak_ptr<ImapContext> ImapContextWeakPtr;

inline ShortSessionInfo createShortSessionInfo(ImapContext& context)
{
    return ShortSessionInfo(
        context.uniq_id(),
        context.userData,
        context.sessionInfo.remoteAddress,
        context.sessionInfo.remotePort);
}

TimerPtr createTimer(ImapContextPtr ctx, Duration timeout, VoidFunction hook);
TimerPtr createTimer(ImapContextPtr ctx, Duration timeout);

}
