#include "lang_config.h"

#include <common/imap_context.h>

namespace yimap {

// Default english localization if no special config for given language was found
const LanguageConfig::SymbolMap LanguageConfig::englishMapping{
    { "inbox", FolderNameEntry{ "inbox", "INBOX", "\\Inbox" } },
    { "draft", FolderNameEntry{ "draft", "Drafts", "\\Drafts" } },
    { "spam", FolderNameEntry{ "spam", "Spam", "\\Spam" } },
    { "sent", FolderNameEntry{ "sent", "Sent", "\\Sent" } },
    { "trash", FolderNameEntry{ "trash", "Trash", "\\Trash" } },
    { "outbox", FolderNameEntry{ "outbox", "Outbox", "" } },
};

// Configuration used to show folder names from database as is
const LanguageConfig::SymbolMap LanguageConfig::databaseMapping{
    { "inbox", FolderNameEntry{ "inbox", "INBOX", "\\Inbox" } },
    { "draft", FolderNameEntry{ "draft", "", "\\Drafts" } },
    { "spam", FolderNameEntry{ "spam", "", "\\Spam" } },
    { "sent", FolderNameEntry{ "sent", "", "\\Sent" } },
    { "trash", FolderNameEntry{ "trash", "", "\\Trash" } },
    { "outbox", FolderNameEntry{ "outbox", "", "" } },
};

LanguageConfig::LanguageConfig()
{
    symbolToFolder = englishMapping;
}

LanguageConfig::LanguageConfig(
    const Ptree& xml,
    const UserSettings& userSettings,
    const string& userLanguage)
    : renameEnabled(userSettings.renameEnabled || xml.get("<xmlattr>.force_rename", 0))
    , localize(userSettings.localizeImap || xml.get("<xmlattr>.force_localize", 0))
{
    std::string language;
    if (localize)
    {
        // If "localize" flag is on, we use language from blackbox
        language = userLanguage;
    }
    symbolToFolder = getLanguageMapping(xml, language);
}

LanguageConfig::SymbolMap LanguageConfig::getLanguageMapping(
    const Ptree& xml,
    const std::string& targetLanguage)
{
    SymbolMap result;
    for (auto& foldersNode : xml)
    {
        // We should find target language
        if (targetLanguage != foldersNode.first) continue;

        for (auto& folder : foldersNode.second)
        {
            auto symbol = folder.first;
            auto name = folder.second.get("<xmlattr>.name", "");
            auto xlist = folder.second.get("<xmlattr>.xlist", "");
            result[symbol] = FolderNameEntry{ symbol, name, xlist };
        }
        break;
    }

    // If no lacalization found, we shold use...
    if (result.empty())
    {
        // ...english names in case of localize_imap flag or rename_enabled, or database names
        // otherwise.
        result = (localize || renameEnabled) ? englishMapping : databaseMapping;
    }

    return result;
}

std::string LanguageConfig::xlistFromSymbol(const string& folderSymbol) const
{
    auto found = symbolToFolder.find(folderSymbol);
    if (found != symbolToFolder.end()) return found->second.xlist;
    return "";
}

std::string LanguageConfig::nameFromSymbol(const string& folderSymbol) const
{
    auto found = symbolToFolder.find(folderSymbol);
    if (found != symbolToFolder.end())
    {
        if (folderSymbol == "inbox" || renameEnabled || localize) return found->second.name;
    }
    return "";
}

} // namespace yimap
