#pragma once

#include <common/types.h>
#include <memory>
#include <mutex>
#include <spdlog/spdlog.h>
#include <yplatform/util/safe_call.h>
#include <yplatform/log.h>

namespace yimap {

class SpdLog
{
public:
    struct Settings
    {
        std::string name;
        std::string fileName;
        std::string format;
        bool autoFlush;
        size_t queueSize;
    };

    SpdLog() = default;
    SpdLog(SpdLog&) = delete;
    SpdLog(SpdLog&&) = delete;
    SpdLog& operator=(SpdLog&) = delete;
    SpdLog& operator=(SpdLog&&) = delete;

    ~SpdLog()
    {
        if (logIsInit)
        {
            yplatform::safe_call(spdlog::drop, name);
        }
    }

    void init(const Settings& logSettings)
    {
        static std::once_flag asyncMode;
        std::call_once(asyncMode, setAsyncMode, std::ref(logSettings.queueSize));

        name = logSettings.name;
        fileName = logSettings.fileName;
        format = logSettings.format;
        autoFlush = logSettings.autoFlush;

        internalInit();
    }

    void rotate()
    {
        try
        {
            internalInit();
        }
        catch (const spdlog::spdlog_ex& ex)
        {
            YLOG_GLOBAL(error) << "Rotate error (" << fileName << "): " << ex.what();
        }
    }

    void logString(const std::string& msg)
    {
        try
        {
            if (!logIsInit)
            {
                internalInit();
            }

            loggerPtr->info() << msg;
        }
        catch (const spdlog::spdlog_ex& ex)
        {
            YLOG_GLOBAL(error) << "Log error(" << fileName << "): " << ex.what();
        }
    }

protected:
    void internalInit()
    {
        try
        {
            auto tmpPtr = std::make_shared<spdlog::logger>(
                name, std::make_shared<spdlog::sinks::simple_file_sink_mt>(fileName, autoFlush));
            spdlog::drop(name);
            spdlog::register_logger(tmpPtr);
            tmpPtr->set_pattern(format.empty() ? "[%Y-%m-%d %H:%M:%S.%e] %v" : format);
            loggerPtr = tmpPtr;
            logIsInit = true;
        }
        catch (const spdlog::spdlog_ex& ex)
        {
            spdlog::drop(name);
            logIsInit = false;
            throw;
        }
    }

    static void setAsyncMode(const size_t queueSize)
    {
        try
        {
            spdlog::set_async_mode(queueSize);
        }
        catch (const spdlog::spdlog_ex& ex)
        {
            throw std::runtime_error("Invalid queueSize for spdLog");
        }
    }

    std::string name;
    std::string fileName;
    std::string format;
    bool autoFlush = true;

    std::shared_ptr<spdlog::logger> loggerPtr;
    std::atomic<bool> logIsInit = { false };
};

class ClientsSpdLogFactory
{
public:
    using Logger = SpdLog;

    ClientsSpdLogFactory(const Ptree& logSettings)
        : logsDir(logSettings.get("log_settings.clients_log.dir", ""))
        , format(logSettings.get("log_settings.clients_log.format", ""))
        , autoFlush(logSettings.get("log_settings.clients_log.auto_flush", 1))
        , queueSize(logSettings.get("log_settings.clients_log.queue_size", 1024 * 1024))
    {
        if (logsDir.empty())
        {
            throw std::runtime_error("Not specified clients log dir");
        }
    }

    std::shared_ptr<Logger> create(const std::string& uid)
    {
        Logger::Settings settings;
        settings.name = uid;
        settings.fileName = logsDir + "/" + uid + ".log";
        settings.format = format;
        settings.autoFlush = autoFlush;
        settings.queueSize = queueSize;

        auto logger = std::make_shared<Logger>();
        logger->init(settings);
        return logger;
    }

private:
    std::string logsDir;
    std::string format;
    bool autoFlush;
    size_t queueSize;
};

template <class LoggerFactory>
class ClientsLogPool
{
public:
    using Logger = typename LoggerFactory::Logger;
    typedef std::shared_ptr<Logger> LoggerPtr;
    typedef std::shared_ptr<LoggerFactory> LoggerFactoryPtr;

    ClientsLogPool() : isInit(false)
    {
    }

    void init(size_t maxPoolSize, LoggerFactoryPtr factory)
    {
        if (!factory)
        {
            throw std::runtime_error("Factory cannot be empty");
        }
        std::lock_guard<std::mutex> lock(mtx);
        this->maxPoolSize = maxPoolSize;
        this->factory = factory;
        this->isInit = true;
    }

    LoggerPtr openSession(const std::string& uid)
    {
        std::lock_guard<std::mutex> lock(mtx);
        if (!isInit)
        {
            return LoggerPtr();
        }
        auto it = loggers.find(uid);
        if (it == loggers.end())
        { // if there is no alredy open logger, then create new one
            if (loggers.size() >= maxPoolSize)
            {
                YLOG_GLOBAL(info) << "Cannot open client log (uid=" << uid << "). Limit exceeded";
                return LoggerPtr();
            }
            it = loggers.emplace(uid, std::make_pair(0, factory->create(uid))).first;
        }
        ++it->second.first; // increment counter
        return it->second.second;
    }

    void closeSession(const std::string& uid)
    {
        std::lock_guard<std::mutex> lock(mtx);
        if (!isInit)
        {
            return;
        }
        auto it = loggers.find(uid);
        if (it != loggers.end())
        {
            --it->second.first; // decrement counter
            if (it->second.first <= 0)
            {
                // if nobody else dont use this logger, then remove it
                loggers.erase(it);
            }
        }
    }

    void rotate()
    {
        std::unique_lock<std::mutex> lock(mtx);
        if (!isInit)
        {
            return;
        }
        std::vector<LoggerPtr> loggersRotate;
        loggersRotate.reserve(loggers.size());
        for (auto it : loggers)
        {
            loggersRotate.push_back(it.second.second);
        }
        lock.unlock();
        for (auto logger : loggersRotate)
        {
            logger->rotate();
        }
    }

private:
    size_t maxPoolSize;
    std::map<std::string, std::pair<int /*counter*/, LoggerPtr>> loggers;
    std::mutex mtx;
    LoggerFactoryPtr factory;
    bool isInit;
};

typedef ClientsLogPool<ClientsSpdLogFactory> ClientsSpdLogPool;

class ClientsSpdLogRotate
{
public:
    ClientsSpdLogRotate(ClientsSpdLogPool& pool) : pool(pool)
    {
    }

    void rotate()
    {
        pool.rotate();
    }

private:
    ClientsSpdLogPool& pool;
};

} // namespace yimap
