#include "session_logger.h"

#include <common/log/log.h>
#include <common/log/clients_log.hpp>

namespace yimap {

SessionLogger::SessionLogger(
    const ShortSessionInfo& sessionInfo,
    const ImapLogSettings& logSettings,
    const ClientIpHints& hints)
    : errorLog(YGLOBAL_LOG_SERVICE, "error_log")
    , warningLog(YGLOBAL_LOG_SERVICE, "warning_log")
    , timingLog(YGLOBAL_LOG_SERVICE, "timing_log")
    , trafficLog(YGLOBAL_LOG_SERVICE, "traffic_log")
    , clientLogEnabled(false)
{
    reset(sessionInfo, logSettings, hints);
}

SessionLogger::~SessionLogger()
{
    flushStash();
    if (clientLogPtr)
    {
        clientsLogPool.closeSession(sessionInfo.userData.uid);
    }
}

void SessionLogger::enableClientLog()
{
    Lock lock(mux);
    clientLogEnabled = true;
    createLoggerIfNeeded();
}

void SessionLogger::createLoggerIfNeeded()
{
    // either enabled in user settings or forced for all users in config
    const bool enabled = clientLogEnabled || logSettings.forceClientsLog;

    if (!clientLogPtr && !sessionInfo.userData.uid.empty() && enabled)
    {
        try
        {
            clientLogPtr = clientsLogPool.openSession(sessionInfo.userData.uid);
        }
        catch (std::exception& e)
        {
            logError() << "Cannot create clientLog: " << e.what();
        }
    }
}

void SessionLogger::reset(
    const ShortSessionInfo& sessionInfo,
    const ImapLogSettings& logSettings,
    const ClientIpHints& hints)
{
    reset(sessionInfo);
    this->logSettings = logSettings;
    this->clientIpHints = hints;
    createLoggerIfNeeded();
}

void SessionLogger::reset(const ShortSessionInfo& sessionInfo)
{
    this->sessionInfo = sessionInfo;
}

void SessionLogger::logString(const string& message, size_t flags)
{
    if (clientIpHints.clientLog) clientLog(message);

    if (flags & LogHelper::LH_LOG_SYSTEM)
    {
        YLOG_GLOBAL(info) << sessionInfo.logPrefix() << message;
    }

    if (flags & LogHelper::LH_LOG_ERROR)
    {
        YLOG(errorLog, error) << sessionInfo.logPrefix() << message;
    }

    if (flags & LogHelper::LH_LOG_WARNING)
    {
        YLOG(warningLog, warning) << sessionInfo.logPrefix() << message;
    }
}

LogHelper SessionLogger::log(bool systemLog, bool errorLog)
{
    return LogHelper(*this, "", systemLog, errorLog);
}

LogHelper SessionLogger::logDebug()
{
    return LogHelper(
        "debug: ",
        (clientIpHints.debugLog ? LogHelper::LH_LOG_DEBUG : LogHelper::LH_LOG_DISCARD),
        *this);
}

LogHelper SessionLogger::logDebugNet()
{
    return LogHelper(
        "debug_net: ",
        (clientIpHints.networkLog ? LogHelper::LH_LOG_DEBUG : LogHelper::LH_LOG_DISCARD),
        *this);
}

void SessionLogger::logTiming(const string& message)
{
    if (logSettings.timingLogEnable)
    {
        YLOG(timingLog, info) << sessionInfo.logPrefix() << message;
    }
}

void SessionLogger::logTraffic(const string& message)
{
    YLOG(trafficLog, debug) << sessionInfo.logPrefix() << message;
}

void SessionLogger::doClientLog(const std::string& message) const
{
    if (clientLogPtr)
    {
        clientLogPtr->logString(message);
    }
}

void SessionLogger::stashRecord(std::string message) const
{
    static const size_t MAX_LOG_STASH = 16;
    if (recordsStash.size() < MAX_LOG_STASH)
    {
        recordsStash.emplace_back(std::move(message));
    }
}

void SessionLogger::flushStash() const
{
    for (const auto& message : recordsStash)
    {
        doClientLog(message);
    }
    recordsStash.clear();
}

} // namespace yimap
