#pragma once

#include <common/log/logger.h>
#include <common/log/clients_log.hpp>
#include <common/settings.h>

#include <yplatform/log.h>
#include <yplatform/util.h>

namespace yimap {

class SessionLogger : public Logger
{
    using Lock = std::lock_guard<std::mutex>;

public:
    SessionLogger(
        const ShortSessionInfo& sessionInfo,
        const ImapLogSettings& logSettings,
        const ClientIpHints& hints);

    ~SessionLogger();

    void reset(
        const ShortSessionInfo& sessionInfo,
        const ImapLogSettings& logSettings,
        const ClientIpHints& hints);

    void reset(const ShortSessionInfo& sessionInfo);

    virtual void updateSessionInfo(const ShortSessionInfo& newSessionInfo) override
    {
        sessionInfo = newSessionInfo;
        createLoggerIfNeeded();
    }

    virtual void enableClientLog() override;

    bool shouldLogCommands() const
    {
        return !logSettings.log_network_data && clientIpHints.clientLog;
    }
    bool shouldLogNetwork() const
    {
        return logSettings.log_network_data && clientIpHints.clientLog;
    }

    // Logging
    virtual void logString(const string&, size_t flags = 0);
    virtual LogHelper log(bool systemLog = false, bool errorLog = false);
    virtual LogHelper logDebug();
    virtual LogHelper logDebugNet();

    virtual void logTiming(const string& message);
    virtual void logCommand(const zerocopy::SegmentRange&) const;
    virtual void logTraffic(const string& message);

    template <typename IterT>
    void logCommandResponse(boost::iterator_range<IterT> resp, const string& debugRecord) const;

    template <typename IterT>
    void logRawData(boost::iterator_range<IterT> data, bool in) const;

protected:
    vector<string> directionPrefix = { "<-| ", "->| ", "" };

    enum ClientLogDirection
    {
        CLD_Client = 0,
        CLD_Server = 1,
        CLD_Info = 2
    };

    void createLoggerIfNeeded();

    template <typename Record>
    void clientLog(const Record& record, ClientLogDirection direction = CLD_Info) const
    {
        Lock lock(mux);
        ostringstream stream;
        stream << boost::posix_time::microsec_clock::local_time() << " <" << sessionInfo.clientPort
               << ">"
               << " [" << sessionInfo.userData.login << "] " << directionPrefix[direction]
               << record;

        if (!clientLogPtr)
        {
            stashRecord(stream.str());
        }
        else
        {
            flushStash();
            doClientLog(stream.str());
        }
    }

    void doClientLog(const std::string& message) const;
    void stashRecord(std::string message) const;
    void flushStash() const;

protected:
    ShortSessionInfo sessionInfo;
    ImapLogSettings logSettings;
    ClientIpHints clientIpHints;

    yplatform::log::source errorLog;
    yplatform::log::source warningLog;
    yplatform::log::source timingLog;
    yplatform::log::source trafficLog;
    std::shared_ptr<SpdLog> clientLogPtr;
    bool clientLogEnabled;

    // We should keep some log records before user login
    mutable std::vector<std::string> recordsStash;
    mutable std::mutex mux;
};

namespace {
const std::locale& myloc = std::locale();
}

inline void SessionLogger::logCommand(const zerocopy::SegmentRange& cmd) const
{
    if (shouldLogCommands())
    {
        try
        {
            std::ostringstream os;
            if (boost::icontains(cmd, " login ", myloc))
            {
                os << " LOGIN COMMAND";
            }
            else
            {
                yplatform::util::log_cmd(os, cmd, logSettings.max_log_chunk);
            }
            clientLog(os.str(), CLD_Client);
        }
        catch (...)
        {
        }
    }
}

template <typename IterT>
void SessionLogger::logCommandResponse(boost::iterator_range<IterT> resp, const string& debugRecord)
    const
{
    if (shouldLogCommands())
    {
        try
        {
            std::ostringstream os;
            yplatform::util::log_cmd(os, resp, logSettings.max_log_chunk);
            string logRecord = os.str();

            if (!debugRecord.empty())
            {
                // log_cmd — removes last \n
                auto endlpos = logRecord.rfind('\r');
                if (endlpos != string::npos && endlpos == logRecord.length() - 1)
                {
                    logRecord.insert(endlpos, debugRecord);
                }
            }
            clientLog(logRecord, CLD_Server);
        }
        catch (...)
        {
        }
    }
}

template <typename IterT>
void SessionLogger::logRawData(boost::iterator_range<IterT> data, bool in) const
{
    if (clientIpHints.clientLog)
    {
        clientLog(data, (in ? CLD_Client : CLD_Server));
    }
}

} // namespace yimap
