#include <common/message_list_diff.h>

using namespace boost;

namespace yimap {

MailboxDiff::MailboxDiff(const FolderInfo& aFolderInfo)
    : addedCount(0), initialRevision(aFolderInfo.revision), currentFolderInfo(aFolderInfo)
{
}

MailboxDiff::MailboxDiff(
    const FolderInfo& aFolderInfo,
    const MessageVector& newDeleted,
    const MessageVector& newChanged,
    const MessageVector& added)
    : addedCount(0), initialRevision(aFolderInfo.revision), currentFolderInfo(aFolderInfo)
{
    for (size_t i = 0; i < newDeleted.size(); i++)
    {
        addDeleted(newDeleted[i]);
    }
    for (size_t i = 0; i < newChanged.size(); i++)
    {
        addChanged(newChanged[i]);
    }
    for (size_t i = 0; i < added.size(); i++)
    {
        addAdded(added[i]);
    }
}

void MailboxDiff::drop(const FolderInfo& folderInfo)
{
    history.clear();
    changed.clear();
    deleted.clear();
    addedCount = 0;
    initialRevision = folderInfo.revision;
    currentFolderInfo = folderInfo;
}

// Add  all changes from other, with revision greater than current revision.
void MailboxDiff::update(MailboxDiff& other)
{
    uint32_t currentRevision = revision();
    for (const MessageData& msg : other.history)
    {
        if (msg.modseq + 1 <= currentRevision) continue;
        addMessage(msg);
    }
}

void MailboxDiff::setFolderInfo(const FolderInfo& folderInfo)
{
    if (folderInfo.revision >= currentFolderInfo.revision)
    {
        bool recentChanged = currentFolderInfo.recentCount != folderInfo.recentCount;
        currentFolderInfo = folderInfo;
        currentFolderInfo.recentChanged = recentChanged;
    }
}

void MailboxDiff::addMessage(const MessageData& msg)
{
    // Message could not be added and deleted at the same time
    assert(!(msg.added && msg.deleted));
    if (msg.deleted)
    {
        addDeleted(msg);
    }
    else if (msg.added)
    {
        addAdded(msg);
    }
    else
    {
        addChanged(msg);
    }
}

void MailboxDiff::addDeleted(const MessageData& msg)
{
    MessageSet::iterator existInChanged = changed.find(msg);
    MessageSet::iterator existInDeleted = deleted.find(msg);
    if (existInChanged != changed.end())
    {
        changed.erase(existInChanged);
    }
    if (existInDeleted == deleted.end())
    {
        if (currentFolderInfo.messageCount > 0)
        {
            currentFolderInfo.messageCount--;
        }
        else
        {
            // TODO: log error?
        }

        auto res = deleted.insert(msg);
        res.first->deleted = true;
        res.first->added = false;

        currentFolderInfo.revision = std::max(currentFolderInfo.revision, msg.modseq + 1);
        history.insert(*res.first);
    }
}

void MailboxDiff::addAdded(const MessageData& msg)
{
    MessageSet::iterator existInChanged = changed.find(msg);
    MessageSet::iterator existInDeleted = deleted.find(msg);

    if (existInChanged == changed.end() && existInDeleted == deleted.end())
    {
        auto res = changed.insert(msg);
        res.first->added = true;
        res.first->deleted = false;

        currentFolderInfo.messageCount++;
        currentFolderInfo.revision = std::max(currentFolderInfo.revision, msg.modseq + 1);
        addedCount++;

        history.insert(*res.first);
    }
}

void MailboxDiff::addChanged(const MessageData& msg)
{
    if (msg.deleted)
    {
        addDeleted(msg);
        return;
    }
    if (msg.added)
    {
        addAdded(msg);
        return;
    }

    MessageSet::iterator existInDeleted = deleted.find(msg);
    MessageSet::iterator existInChanged = changed.find(msg);
    // If message was deleted or are newer than existed we shoud erase it
    if ((existInDeleted != deleted.end()) ||
        (existInChanged != changed.end() && msg.modseq > existInChanged->modseq))
    {
        changed.erase(msg);
    }
    // If this message was not deleted, we should insert back its newer version.
    if (existInDeleted == deleted.end())
    {
        changed.insert(msg);
    }
    currentFolderInfo.revision = std::max(currentFolderInfo.revision, msg.modseq + 1);
    history.insert(msg);
}

}
