#pragma once

#include <common/flags.h>
#include <common/types.h>
#include <set>
#include <limits>

#include <macs/mime_part.h>

#include <boost/cstdint.hpp> // uint32_t
#include <boost/shared_ptr.hpp>

namespace yimap {

using MidSet = std::set<uint64_t>;

struct MessageExtra
{
    MessageExtra() = default;
    MessageExtra(const string& mid, uint32_t size) : mid(mid), size(size)
    {
    }

    string mid;
    uint32_t size = 0;

    const string& smid() const
    {
        return mid;
    }
    void smid(const string& newMid)
    {
        mid = newMid;
    }

    bool partial() const
    {
        return size == 0;
    }
    operator bool() const
    {
        return !partial();
    }
};

struct MessageDetails : public MessageExtra
{
    MessageDetails() = default;
    MessageDetails(const MessageDetails& other) : MessageExtra(other), uid(other.uid)
    {
    }
    explicit MessageDetails(uint32_t anUid) : MessageExtra(), uid(anUid)
    {
    }
    MessageDetails(uint32_t anUid, const MessageExtra& extra) : MessageExtra(extra), uid(anUid)
    {
    }

    MessageDetails(uint32_t anUid, const string& mid, uint32_t size)
        : MessageExtra(mid, size), uid(anUid)
    {
    }

    MessageDetails(MessageDetails&& that) : MessageExtra(std::move(that)), uid(that.uid)
    {
    }
    MessageDetails& operator=(const MessageDetails& that)
    {
        if (this != &that)
        {
            mid = that.mid;
            size = that.size;
            uid = that.uid;
        }

        return *this;
    }

    uint32_t uid = 0;
};

using Revision = std::size_t;

struct MessageData
{
    MessageData() = default;
    MessageData(const MessageData&) = default;
    MessageData& operator=(const MessageData&) = default;
    explicit MessageData(uint32_t uid, uint64_t mid = 0) : uid(uid), mid(mid)
    {
    }
    ~MessageData()
    {
    }

    void setDetails(const string& mid, time_t aTime, uint32_t size);

    uint32_t num = 0; // imap number
    uint32_t uid = 0; // imap id
    uint64_t mid = 0;
    uint32_t modseq = 0;
    time_t time = 0;
    mutable Flags flags;

    mutable bool recent = false;
    mutable bool added = false;
    mutable uint32_t deleted = 0;

    mutable uint32_t baseUid = 0;
    mutable uint32_t offset = 0;

    MessageDetails details;

    template <typename RangeT>
    bool has_flag(RangeT const& fl) const
    {
        return flags.hasFlag(fl);
    }
    bool has_flag(Flags::MessageFlag const& fl) const
    {
        return flags.hasFlag(fl);
    }
    bool operator<(const MessageData& that) const
    {
        return this->uid < that.uid;
    }

    const std::string smid() const
    {
        return mid == 0 ? (details.mid) : std::to_string(mid);
    }

    static uint64_t takeMid(const MessageData& m)
    {
        return m.mid;
    }

    using Predicate = std::function<bool(const MessageData&)>;

    static bool id(const MessageData&)
    {
        return true;
    }
    static bool isPartial(const MessageData& m)
    {
        return m.details.partial();
    }

    std::string dump() const;
};

inline void MessageData::setDetails(const string& mid, time_t aTime, uint32_t size)
{
    details = MessageDetails(uid, mid, size);
    time = aTime;
}

inline std::string MessageData::dump() const
{
    std::ostringstream os;

    os << "num=" << num << " "
       << "uid=" << uid << " "
       << "mid=" << mid << " "
       << "baseUid=" << baseUid << " "
       << "offset=" << offset << " "
       << "recent=" << recent << " "
       << "added=" << added << " "
       << "deleted=" << deleted << " ";

    return os.str();
}

struct BodyMetadata
{
    std::string stid;
    macs::MimeParts mimeParts;
};

using BodyMetadataByMid = std::map<std::string, BodyMetadata>;

} // namespace yimap
