#pragma once

#include <common/message_data.h>

namespace yimap {

class MessageVector : public std::vector<MessageData>
{
public:
    MessageVector() = default;
    MessageVector(const MessageVector&) = default;
    MessageVector& operator=(const MessageVector&) = default;

    void sortByRevision();
    void sortByUid();

    MidSet asMidSet() const;
};

typedef MessageVector MessagesVector;
typedef std::shared_ptr<MessagesVector> MessagesVectorPtr;

///////////////////////////////////////////////////////////////////////////////

struct CompareMsgByUid
{
    bool operator()(const MessageData& lhs, const MessageData& rhs) const
    {
        return lhs.uid < rhs.uid;
    }
};

class MessageSet : public std::set<MessageData, CompareMsgByUid>
{
public:
    MessageSet() = default;
    MessageSet(const MessageSet&) = default;
    MessageSet& operator=(const MessageSet&) = default;
};

struct CompareByRevision
{
    bool operator()(const MessageData& lhs, const MessageData& rhs) const
    {
        return lhs.modseq < rhs.modseq;
    }
};
typedef std::multiset<MessageData, CompareByRevision> MessageMultiSet;

struct CompareByRevisionAndUid
{
    bool operator()(const MessageData& lhs, const MessageData& rhs) const
    {
        if (lhs.modseq == rhs.modseq)
        {
            return lhs.uid < rhs.uid;
        }
        return lhs.modseq < rhs.modseq;
    }
};
// Set, sorted by revision first, and by uid second.
typedef std::set<MessageData, CompareByRevisionAndUid> MessageRevisionedSet;
typedef std::set<MessageData, CompareByRevisionAndUid> MessageSetByRevision;

//-----------------------------------------------------------------------------
// Set of message details like stid or size

struct CompareDetailsByUid
{
    bool operator()(const MessageDetails& lhs, const MessageDetails& rhs) const
    {
        return lhs.uid < rhs.uid;
    }
};

typedef std::set<MessageDetails, CompareDetailsByUid> MessageDetailsSet;
typedef boost::shared_ptr<MessageDetailsSet> MessageDetailsSetPtr;

typedef std::vector<MessageDetails> MessageDetailsVector;
typedef boost::shared_ptr<MessageDetailsVector> MessageDetailsVectorPtr;

//-----------------------------------------------------------------------------
// Mid list

struct SmidList : public std::list<std::string>
{
    SmidList()
    {
    }
    SmidList(const MessageVector& messages)
    {
        for (auto msg : messages)
        {
            push_back(msg.smid());
        }
    }
};
typedef std::shared_ptr<SmidList> SmidListPtr;

} // namespace yimap
