#pragma once

#include <common/types.h>
#include <common/errors.h>
#include <common/message_data.h>

#include <list>
#include <stdexcept>

namespace yimap {

typedef std::pair<unsigned int, unsigned int> range_t;

namespace {
const range_t empty_range(0, 0);
}

inline std::ostream& operator<<(std::ostream& os, const range_t& r)
{
    os << "[" << r.first << "," << r.second << "]";
    return os;
}

inline bool is_intersect(const range_t& r1, const range_t& r2)
{
    return (
        (r1.first >= r2.first - 1 && r1.first <= r2.second + 1) ||
        (r1.second >= r2.first - 1 && r1.second <= r2.second + 1));
}

inline range_t operator+(const range_t& r1, const range_t& r2)
{
    return range_t(std::min(r1.first, r2.first), std::max(r1.second, r2.second));
}

template <class SeqIterator>
class SeqRangeIterator
    : public boost::iterator_facade<
          SeqRangeIterator<SeqIterator>,
          unsigned int,
          boost::bidirectional_traversal_tag,
          unsigned int>
{
    struct enabler
    {
    };

public:
    inline SeqRangeIterator(void)
    {
    }
    explicit inline SeqRangeIterator(const SeqIterator& b, const SeqIterator& e = SeqIterator())
        : b_(b), e_(e == SeqIterator() ? b : e)
    {
        set_val();
    }

    template <class OtherSeq>
    inline SeqRangeIterator(
        const SeqRangeIterator<OtherSeq>& other,
        typename boost::enable_if<boost::is_convertible<OtherSeq, SeqIterator>, enabler>::type =
            enabler())
        : b_(other.b_), e_(other.e_), val_(other.val_), sign_(other.sign_)
    {
    }

private:
    inline void set_val(void)
    {
        if (b_ != e_)
        {
            val_ = b_->first;
            sign_ = (b_->first < b_->second) ? 1 : -1;
        }
    }

    template <class>
    friend class SeqRangeIterator;
    friend class boost::iterator_core_access;

    inline void increment(void)
    {
        val_ += sign_;

        if ((sign_ > 0 && val_ > b_->second) || (sign_ < 0 && val_ < b_->second))
        {
            ++b_;
            set_val();
        }
    }

    inline void decrement(void)
    {
        val_ -= sign_;

        if ((sign_ > 0 && val_ < b_->first) || (sign_ < 0 && val_ > b_->first))
        {
            --b_;
            set_val();
        }
    }

    typename SeqRangeIterator::iterator_facade_::reference inline dereference(void) const
    {
        return val_;
    }

    template <class OtherSeq>
    inline bool equal(
        const SeqRangeIterator<OtherSeq>& other,
        typename boost::enable_if<boost::is_convertible<OtherSeq, SeqIterator>, enabler>::type =
            enabler()) const
    {
        if (b_ != other.b_) return false;
        if (b_ == e_) return true;
        return (val_ == other.val_);
    }

private:
    SeqIterator b_;
    const SeqIterator e_;
    unsigned int val_;
    short sign_;
};

struct CompareRange
{
    bool operator()(const range_t& lhs, const range_t& rhs) const
    {
        return lhs.second < rhs.second;
    }
};

class seq_range;
using SeqRangePtr = std::shared_ptr<seq_range>;

class seq_range : public std::set<range_t, CompareRange>
{
public:
    typedef SeqRangeIterator<iterator> range_iterator;
    typedef SeqRangeIterator<const_iterator> const_range_iterator;

    seq_range() = default;
    seq_range(unsigned int min_val, unsigned int max_val, bool byUid)
        : uidMode_(byUid), min_val_(min_val), max_val_(max_val)
    {
    }

    bool contains(const MessageData& message) const
    {
        range_t rng(uidMode_ ? message.uid : message.num, uidMode_ ? message.uid : message.num);
        iterator lower = lower_bound(rng);
        // if (lower->first > rng.first && lower != begin()) lower++;
        return lower != end() && rng.first >= lower->first && rng.second <= lower->second;
    }

    seq_range& operator+=(range_t r);
    seq_range& operator+=(const seq_range& other);

    range_iterator begin_range(void)
    {
        return range_iterator(begin(), end());
    }
    range_iterator end_range(void)
    {
        return range_iterator(end(), end());
    }

    const_range_iterator begin_range(void) const
    {
        return const_range_iterator(begin(), end());
    }

    const_range_iterator end_range(void) const
    {
        return const_range_iterator(end(), end());
    }

    range_t Union() const
    {
        return union_;
    }

    unsigned int minVal() const
    {
        return min_val_;
    }
    unsigned int maxVal() const
    {
        return max_val_;
    }

    unsigned int first() const
    {
        return begin()->first;
    }
    unsigned int last() const
    {
        return rbegin()->second;
    }

    bool uidMode() const
    {
        return uidMode_;
    }

private:
    bool uidMode_ = false;
    unsigned int min_val_ = 0;
    unsigned int max_val_ = 0;
    range_t union_{ 0, 0 };
};

inline seq_range& seq_range::operator+=(range_t r)
{
    if (r != empty_range)
    {
        iterator b = begin();

        while (b != end() && !(r.second < b->first - 1))
        {
            if (is_intersect(r, *b))
            {
                r = (*b) + r;
                erase(b);
                b = begin();
            }
            else
                ++b;
        }

        // push_back ( r );
        insert(b, r);
        if (union_ == empty_range)
        {
            union_ = r;
        }
        else
        {
            union_ = union_ + r;
        }
    }
    return *this;
}

inline seq_range& seq_range::operator+=(const seq_range& other)
{
    for (const range_t& rng : other)
    {
        *this += rng;
    }
    return *this;
}

inline std::ostream& operator<<(std::ostream& os, const seq_range& seq)
{
    for (seq_range::const_iterator rng = seq.begin(); rng != seq.end(); rng++)
    {
        os << *rng;
    }
    return os;
}

inline seq_range trim_left_copy(const seq_range& range, size_t newBorder)
{
    auto res = seq_range(range.minVal(), range.maxVal(), range.uidMode());

    auto it = range.upper_bound({ 0, newBorder });
    if (it == range.end())
    {
        return res;
    }

    if (it->first <= newBorder)
    {
        auto range = *it;
        range.first = newBorder + 1;
        res += range;
        ++it;
    }
    res.insert(it, range.end());
    return res;
}

inline seq_range trim_right_copy(const seq_range& range, size_t newBorder)
{
    auto it = range.lower_bound({ 0, newBorder });
    if (it == range.end())
    {
        return range;
    }

    auto res = seq_range(range.minVal(), range.maxVal(), range.uidMode());
    if (it->first < newBorder)
    {
        auto val = *it;
        val.second = newBorder - 1;
        res += val;
    }
    res.insert(range.begin(), it);
    return res;
}

} // namespace
