#pragma once

#include <common/errors.h>
#include <common/types.h>
#include <common/seq_range.h>
#include <common/lex_ids.h>

#include <stdexcept>
#include <boost/variant.hpp>
#include <boost/exception/all.hpp>
#include <boost/iterator/iterator_facade.hpp>

namespace yimap {

template <class TREE>
range_t makeRange(const TREE& v, unsigned int min_val, unsigned int max_val)
{
    using namespace lex_ids;

    switch (v.value.id().to_long())
    {
    default:
        throw std::runtime_error("invalid sequence range");

    case SEQ_NUMBER:
    {
        unsigned int val;
        val = boost::get<unsigned int>(v.value.value());
        if (val > max_val || val < min_val) return empty_range;
        return range_t(val, val);
    }

    case SEQ_NUMBER_LAST:
        return range_t(max_val, max_val);

    case SEQ_RANGE:
    {
        if (v.children.size() != 2) throw std::runtime_error("invalid sequence range");
        const typename TREE::children_t& vv = v.children;
        unsigned int p[2];

        bool overflow = false;
        switch (vv[0].value.id().to_long())
        {
        default:
            throw std::runtime_error("invalid sequence range");
        case SEQ_NUMBER:
            p[0] = boost::get<unsigned int>(vv[0].value.value());
            overflow = p[0] > max_val;
            break;

        case SEQ_NUMBER_LAST:
            p[0] = max_val;
            overflow = false;
            break;
        }

        switch (vv[1].value.id().to_long())
        {
        default:
            throw std::runtime_error("invalid sequence range");
        case SEQ_NUMBER:
            p[1] = boost::get<unsigned int>(vv[1].value.value());
            break;

        case SEQ_NUMBER_LAST:
            p[1] = max_val;
            overflow = false;
            break;
        }

        range_t rng(std::min(p[0], p[1]), std::max(p[0], p[1]));
        // Trim left and right bounds
        rng.first = std::max(rng.first, min_val);
        rng.second = std::min(rng.second, max_val);
        if (rng.first > max_val || rng.second < min_val || overflow) return empty_range;
        return rng;
    }
    }
}

template <class TREE>
long parseSeqRange(const TREE& tree, seq_range& seq)
{
    long count = 0;
    for (const typename TREE::children_t::value_type& v : tree.children)
    {
        ++count;
        seq += makeRange(v, seq.minVal(), seq.maxVal());
    }
    return count;
}

template <class TREE>
long parseSeqRangeDeprecated(const TREE& tree, seq_range& seq)
{
    try
    {
        return parseSeqRange(tree, seq);
    }
    catch (...)
    {
        return -1;
    }
}

template <class TREE>
void validateSeqRange(const TREE& tree)
{
    static const unsigned int FAKE_MAX_VAL = 100500;
    seq_range tmp(0, FAKE_MAX_VAL, false);
    if (parseSeqRangeDeprecated(tree, tmp) <= 0)
    {
        throw InvalidSeqSet();
    }
}

} // namespace
