#include "settings.h"
#include <yplatform/util.h>
#include <boost/algorithm/string/split.hpp>

namespace yimap {

template <typename T>
inline T createAndLoad(const Ptree& cfg)
{
    T t;
    t.parse_ptree(cfg);
    return t;
}

template <typename IpBytes, typename Hint>
bool checkIpMask(const IpBytes& bytes, const Hint& hint)
{
    if (hint.ipBytesUsed != bytes.size()) return false;

    size_t pos = 0;
    size_t maskBits = hint.maskBitsCount;
    while (maskBits > 7)
    {
        if (bytes[pos] != hint.ipBytes[pos])
        {
            return false;
        }

        pos++;
        maskBits -= 8;
    }

    if (maskBits > 0)
    {
        uint8_t byte1 = bytes[pos];
        byte1 >>= (8 - maskBits);

        uint8_t byte2 = hint.ipBytes[pos];
        byte2 >>= (8 - maskBits);

        if (byte1 != byte2) return false;
    }

    return true;
}

//-----------------------------------------------------------------------------
// Helper structures

IpHintsEntry::IpHintsEntry()
{
}

IpHintsEntry::IpHintsEntry(const string& networkMask, const string& loglevel, size_t cacheTtl)
    : networkMask(networkMask), statusCacheTtl(cacheTtl)
{
    calculateNetmask();
    interpretLogLevel(loglevel);
}

void IpHintsEntry::calculateNetmask()
{
    string bitsSuffix = "128";
    string networkIpStr = networkMask;
    size_t slashPos = networkMask.find('/');
    if (slashPos != string::npos)
    {
        bitsSuffix = networkMask.substr(slashPos + 1);
        networkIpStr = networkMask.substr(0, slashPos);
    }

    maskBitsCount = boost::lexical_cast<size_t>(bitsSuffix);
    boost::asio::ip::address ip = boost::asio::ip::address::from_string(networkIpStr);
    if (ip.is_v4())
    {
        auto bytes = ip.to_v4().to_bytes();
        ipBytesUsed = bytes.size();

        std::copy(bytes.begin(), bytes.end(), ipBytes.begin());
    }
    else
    {
        auto bytes = ip.to_v6().to_bytes();
        ipBytesUsed = bytes.size();

        std::copy(bytes.begin(), bytes.end(), ipBytes.begin());
    }
}

void IpHintsEntry::interpretLogLevel(const string& logLevel)
{
    if (yplatform::iequals("nolog", logLevel)) clientLog = debugLog = false;

    if (yplatform::iequals("debug", logLevel)) clientLog = debugLog = true;

    if (yplatform::iequals("debug_net", logLevel)) clientLog = debugLog = networkLog = true;
}

ClientIpHints::ClientIpHints(const IpHintsEntry& source)
    : clientLog(source.clientLog)
    , debugLog(source.debugLog)
    , networkLog(source.networkLog)
    , statusCacheTtl(source.statusCacheTtl)
{
}

AppendSettings AppendSettings::create(const Ptree& append)
{
    AppendSettings settings;

    settings.smtpgatePercent = append.get("smtpgate_percent", 0u);

    settings.smtpHost = append.get("smtp.host", "");
    settings.smtpPort = append.get("smtp.port", 1234);
    settings.smtpTimeout = append.get("smtp.timeout", 10);
    settings.smtpRetryCount = append.get("smtp.retry_count", 2);
    settings.smtpIpv6 = append.get("smtp.ipv6", 0);

    settings.httpHost = append.get("http.host", "");
    settings.httpPort = append.get("http.port", 2443);
    settings.httpRetryCount = append.get("http.retry_count", 2);
    settings.httpRetrySmtp = append.get("http.retry_smtp", 1);

    return settings;
}

bool AppendSettings::shouldUseSmtpgate(const std::string& sessionId) const
{
    std::hash<std::string> strHash;
    auto pseudoRand = strHash(sessionId) % 99;
    return !httpHost.empty() && smtpgatePercent > pseudoRand;
}

SearchSettings SearchSettings::create(const Ptree& conf)
{
    auto throttling = conf.get_child("throttling");
    SearchSettings settings;
    settings.throttling.notThrottledRequestsCount = throttling.get(
        "not_throttled_requests_count", settings.throttling.notThrottledRequestsCount);
    settings.throttling.delayAfterLastRequest =
        throttling.get("delay_after_last_request", settings.throttling.delayAfterLastRequest);
    settings.throttling.delayAfterPenultRequest =
        throttling.get("delay_after_penult_request", settings.throttling.delayAfterPenultRequest);
    return settings;
}

Settings::Settings(const Ptree& config)
{
    serverSettings = createAndLoad<ServerSettings>(config);
    appendSettings = AppendSettings::create(config.get_child("append"));
    searchSettings = SearchSettings::create(config.get_child("search"));
    maxDBChunkSize = config.get("max_db_chunk_size", maxDBChunkSize);
    idleUpdateTimeout = config.get("append_update_timeout", idleUpdateTimeout);
    enableBBBlock = config.get("enable_bb_block", enableBBBlock);
    languageOptionsXml = config.get_child("language_options");
    allowInboxSubfolders = config.get("allow_inbox_subfolders", allowInboxSubfolders);
    dropFreshCounter = config.get("drop_fresh_counter", dropFreshCounter);
    checkKarma = config.get("check_karma", checkKarma);
    oauthScope = createOauthScope(config);
    maxDetailsLoadingChunk = config.get("max_details_loading_chunk", maxDetailsLoadingChunk);
    moveExtension = config.get("move_extension", moveExtension);
    startTLSExtension = config.get("starttls_extension", startTLSExtension);
    oauthExtension = config.get("oauth_extension", oauthExtension);
    idleExtension = config.get("idle_extension", idleExtension);
    notifyExtension = config.get("notify_extension", notifyExtension);
    reportSpam = config.get_child("report_spam");
    logSettings = ImapLogSettings(config.get_child("log_settings"));
    loginSettings = LoginSettings(config.get_child("login_settings"));
    allowZombie = config.get("allow_zombie", "yes");
    maxBadCommands = config.get("max_bad_commands", maxBadCommands);
    useSettingsRepository = config.get("use_settings_repository", useSettingsRepository);
    xivaAtLogin = config.get("xiva_at_login", xivaAtLogin);
    authRetry = config.get("auth_retry", authRetry);
    authRetryTimeout = config.get("auth_retry_timeout", authRetryTimeout);
    pgUidsAll = config.get("pg_uids.<xmlattr>.all", 0);
    pgUids = readStringSet(config, "pg_uids", "");
    ipHints = readIpHints(config.get_child_optional("ip_hints"));
    defaultDomain = config.get("default_domain", "yandex.ru");
    searchTreeMaxHeight = config.get("search_tree_max_height", searchTreeMaxHeight);
    imapFoldersEncoding_ = config.get("imap_folders_encoding", "utf-7-imap");
    fetchChunkSize = config.get("fetch_chunk_size", fetchChunkSize);
    if (auto throttlingConf = config.get_child_optional("throttling"))
    {
        throttlingSettings = createAndLoad<ThrottlingSettings>(*throttlingConf);
    }

    createRequestMap(config);
    createRawHeadersList(config);
}

void Settings::createRequestMap(const Ptree& config)
{
    PtreeConstOpt map = config.get_child_optional("request_map");
    if (map)
    {
        for (Ptree::const_iterator i = map->begin(); i != map->end(); ++i)
        {
            L_(debug) << "set request_map entry '" << i->first << "' -> '"
                      << i->second.get_value(string("@") + i->first) << "'";
            request_map_[i->first] = i->second.get_value(string("@") + i->first);
        }
    }
}

void Settings::createRawHeadersList(const Ptree& config)
{
    PtreeConstOpt list = config.get_child_optional("raw_value_headers");
    if (list)
    {
        for (auto& entry : *list)
        {
            rawHeadersList.push_back(entry.second.get_value<string>());
        }
    }
}

set<string> Settings::readStringSet(const Ptree& config, const string& group, const string& child)
    const
{
    set<string> result;
    PtreeConstOpt eventsXml = config.get_child_optional(group);
    if (!eventsXml) return result;

    for (Ptree::const_iterator event = eventsXml->begin(); event != eventsXml->end(); event++)
    {
        PtreeConstOpt operationAttr = event->second.get_child_optional(child);
        if (operationAttr) result.insert(operationAttr->get_value(""));
    }

    return result;
}

IpHintsVector Settings::readIpHints(const PtreeConstOpt& ipHintsXml) const
{
    IpHintsVector result;
    if (ipHintsXml)
    {
        std::string defaultLogLevel = ipHintsXml->get("log_level", "");
        for (const auto& ipHintsEntry : *ipHintsXml)
        {
            std::string logLevel = ipHintsEntry.second.get("log_level", defaultLogLevel);
            size_t cacheTtl = ipHintsEntry.second.get("cache_ttl", 0u);
            for (const auto& netmaskEntry : ipHintsEntry.second)
            {
                if (netmaskEntry.first == "netmask")
                {
                    std::string netmask = netmaskEntry.second.get_value("");
                    result.push_back(IpHintsEntry(netmask, logLevel, cacheTtl));
                }
            }
        }
    }
    return result;
}

ClientIpHints Settings::getIpHints(const string& clientIpStr) const
{
    try
    {
        boost::asio::ip::address ip = boost::asio::ip::address::from_string(clientIpStr);
        if (ip.is_v4())
        {
            auto clientIp = ip.to_v4();
            for (auto hint : ipHints)
            {
                if (checkIpMask(clientIp.to_bytes(), hint))
                {
                    return hint;
                }
            }
        }
        else if (ip.is_v6())
        {
            auto clientIp = ip.to_v6();
            for (auto hint : ipHints)
            {
                if (checkIpMask(clientIp.to_bytes(), hint))
                {
                    return hint;
                }
            }
        }
        else
        {
            throw std::runtime_error("Not IPv4 && not IPv6 addres specified");
        }
    }
    catch (std::exception& e)
    {
        L_(debug) << "options::getIpHints for ip " << clientIpStr
                  << " got std::exception: " << e.what();
    }
    return ClientIpHints();
}

size_t Settings::getStatusCacheTtl(const string& clientIp) const
{
    return getIpHints(clientIp).statusCacheTtl;
}

set<string> Settings::createOauthScope(const Ptree& config) const
{
    set<string> result;
    PtreeConstOpt oauthScope = config.get_child_optional("oauth_scope");
    if (!oauthScope) return result;

    for (Ptree::const_iterator i = oauthScope->begin(); i != oauthScope->end(); i++)
    {
        string scope = i->second.get_value<string>("");
        result.insert(scope);
    }
    return result;
}

ReportSpamSettings::ReportSpamSettings(const Ptree& config)
{
    auto spamFlagsSrc = config.get("spam_flags", "");
    auto notSpamFlagsSrc = config.get("notspam_flags", "");
    boost::split(spamFlags, spamFlagsSrc, boost::is_any_of(" "));
    boost::split(notSpamFlags, notSpamFlagsSrc, boost::is_any_of(" "));
}

} // namespace yimap
