#pragma once

#include <map>
#include <string>

#include <common/lang_config.h>
#include <common/helpers/helpers.h>
#include <yplatform/net/settings.h>
#include <yplatform/net/ssl_settings.h>
#include <boost/unordered_map.hpp>

namespace yimap {

using namespace std;

typedef boost::optional<Ptree&> PtreeOpt;
typedef boost::optional<const Ptree&> PtreeConstOpt;

struct AuthSettings
{
    std::string service;
    std::string domain;

    void parse_ptree(const Ptree& data)
    {
        service = data.get("service", "pop");
        domain = data.get("domain", "yandex.ru");
    }
};

struct IpHintsEntry
{
    IpHintsEntry();
    IpHintsEntry(const std::string& networkMask, const std::string& loglevel, size_t cacheTtl);

    std::string networkMask = "::/128";
    // Bits shift value from network mask record "<networkIp>/<networkBitsShift>"
    size_t maskBitsCount = 0;
    // Network IP representation via bits, shifted by networkBitsShift bits from
    // "<networkIp>/<networkBitsShift>"
    std::array<uint8_t, 16> ipBytes;
    size_t ipBytesUsed = 0;

    bool clientLog = true;
    bool debugLog = false;
    bool networkLog = false;

    size_t statusCacheTtl = 0;

private:
    void calculateNetmask();
    void interpretLogLevel(const std::string& logLevel);
};

using IpHintsVector = vector<IpHintsEntry>;

struct ClientIpHints
{
    ClientIpHints() = default;
    ClientIpHints(const IpHintsEntry& source);

    bool clientLog = true;
    bool debugLog = false;
    bool networkLog = false;
    size_t statusCacheTtl = 0;
};

struct ImapLogSettings
{
    bool timingLogEnable = true;
    bool log_network_data = false;
    size_t max_log_chunk = 1024;
    bool forceClientsLog = false;

    ImapLogSettings() = default;
    explicit ImapLogSettings(const Ptree& logConfig)
        : timingLogEnable(logConfig.get("timing_log_enable", timingLogEnable))
        , log_network_data(logConfig.get("log_network_data", 0))
        , max_log_chunk(logConfig.get("max_log_chunk", 1024))
        , forceClientsLog(logConfig.get("clients_log.force_enabled", false))
    {
    }
};

struct ReportSpamSettings
{
    ReportSpamSettings() = default;
    ReportSpamSettings(const Ptree& config);

    std::vector<string> spamFlags;
    std::vector<string> notSpamFlags;
};

struct LoginSettings
{
    LoginSettings() = default;
    LoginSettings(const Ptree& config)
        : max_status_retry(config.get("max_status_retry", max_status_retry))
        , imap_activation_timeout(config.get("imap_activation_timeout", imap_activation_timeout))
        , userJournalLogEnable(config.get("user_journal_log_enable", userJournalLogEnable))
    {
    }

    uint32_t max_status_retry = 3;
    uint32_t imap_activation_timeout = 1500;
    bool userJournalLogEnable = false;
};

struct AppendSettings
{
    size_t smtpgatePercent;

    std::string smtpHost;
    uint32_t smtpPort;
    uint32_t smtpTimeout;
    size_t smtpRetryCount;
    bool smtpIpv6;

    std::string httpHost;
    uint32_t httpPort;
    size_t httpRetryCount;
    bool httpRetrySmtp;

    bool shouldUseSmtpgate(const std::string& sessionId) const;

    static AppendSettings create(const Ptree& append);
};

struct ExpungeSettings
{
    uint32_t chunkThreshold;
    uint32_t chunkSize;
    bool chunkEnable;

    static ExpungeSettings create(const Ptree& expunge);
};

struct UpdFlagsSettings
{
    uint32_t chunkThreshold;
    uint32_t chunkSize;
    bool chunkEnable;

    static UpdFlagsSettings create(const Ptree& updFlags);
};

struct SearchSettings
{
    struct
    {
        size_t notThrottledRequestsCount = 10;
        Duration delayAfterLastRequest = Milliseconds(100);
        Duration delayAfterPenultRequest = Seconds(2);
    } throttling;

    static SearchSettings create(const Ptree& conf);
};

struct ServerEndpointSettings
{
    ServerEndpointSettings() = default;
    ServerEndpointSettings(const string& addr, unsigned short port, bool secure = false)
        : addr(addr), port(port), secure(secure)
    {
    }

    void parse_ptree(const Ptree& conf)
    {
        auto addr_optional = conf.get_optional<std::string>("<xmlattr>.addr");
        addr = addr_optional ? addr_optional.value() : conf.get<std::string>("addr", "");
        auto port_optional = conf.get_optional<unsigned short>("<xmlattr>.port");
        port = port_optional ? port_optional.value() : conf.get<unsigned short>("port", port);
        secure = conf.get<bool>("secure", secure);
        socket_settings.parse_ptree(conf);
    }

    string addr;
    unsigned short port = 0;
    bool secure = false;
    yplatform::net::server_settings socket_settings;
};

struct ServerLimits
{
    static const std::size_t KB = yplatform::net::server_settings::KB;
    static const std::size_t MB = yplatform::net::server_settings::MB;

    std::size_t max_all_literal_size = 40 * MB;
    std::size_t max_literal_size = 40 * MB;
    std::size_t max_pure_size = 8 * KB;
    std::size_t max_output_chunk = 10 * KB;
    std::size_t max_auth_data_size = 2 * KB;

    void parse_ptree(const Ptree& data)
    {
        max_all_literal_size = data.get("max_all_literal_size", max_all_literal_size);
        max_literal_size = data.get("max_literal_size", max_literal_size);
        max_output_chunk = data.get("max_output_chunk", max_output_chunk);
        max_pure_size = data.get("max_pure_size", max_pure_size);
        max_auth_data_size = data.get("max_auth_data_size", max_auth_data_size);
    }
};

struct ThrottlingSettings
{
    size_t limit = 1000;
    Duration windowLength = Seconds(1);
    Duration maxDelay = windowLength * 5;

    void parse_ptree(const Ptree& conf)
    {
        limit = conf.get("limit", limit);
        windowLength = conf.get<Duration>("window_length", windowLength);
        maxDelay = conf.get<Duration>("max_delay", maxDelay);
    }
};

struct ServerSettings
{
    AuthSettings acfg;

    ServerLimits limits;

    std::size_t min_read_buffer_chunk_size = 512;
    std::size_t max_read_buffer_chunk_size = 1024 * 1024;
    std::size_t max_write_buffer_size = 256 * 1024 * 1024;

    yplatform::net::time_duration autologout = Duration::max(); // XXX
    yplatform::net::time_duration autologout_init = Duration::max();
    yplatform::net::time_duration autologout_idle = Duration::max();

    std::string host_name = HostId::create().hostName;

    bool trafficLogEnabled = false;

    yplatform::net::ssl_settings ssl;
    std::vector<ServerEndpointSettings> endpoints;

    bool enableAutoExpunge = true;

    void parse_ptree(const Ptree& data)
    {
        boost::optional<const boost::property_tree::ptree&> acfg_data =
            data.get_child_optional("auth_cfg");

        if (acfg_data) acfg.parse_ptree(acfg_data.get());

        min_read_buffer_chunk_size =
            data.get("read_buffer.min_chunk_size", min_read_buffer_chunk_size);
        max_read_buffer_chunk_size =
            data.get("read_buffer.max_chunk_size", max_read_buffer_chunk_size);

        max_write_buffer_size = data.get("write_buffer.max_size", max_write_buffer_size);

        // Default autologout time is 40 sec.
        int autologout_milsec = data.get("autologout", 40000);
        int autologout_init_milsec = data.get("autologout_init", autologout_milsec);
        int autologout_idle_milsec = data.get("autologout_idle", autologout_milsec);

        if (autologout_milsec >= 0) autologout = Milliseconds(autologout_milsec);
        if (autologout_init_milsec >= 0) autologout_init = Milliseconds(autologout_init_milsec);
        if (autologout_idle_milsec >= 0) autologout_idle = Milliseconds(autologout_idle_milsec);

        limits.parse_ptree(data);

        trafficLogEnabled = data.get("traffic_log_enabled", false);

        ssl.parse_ptree(data.get_child("ssl", Ptree()));

        auto&& base_socket_conf = data.get_child("socket", Ptree());
        auto range = data.get_child("endpoints").equal_range("listen");
        for (auto it = range.first; it != range.second; ++it)
        {
            ServerEndpointSettings ep;
            ep.socket_settings.parse_ptree(base_socket_conf);
            ep.parse_ptree(it->second);
            endpoints.push_back(ep);
        }
        enableAutoExpunge = data.get("enable_auto_expunge", enableAutoExpunge);
    }
};

class Settings
{
public:
    Settings() = default;
    Settings(const Ptree& config);

    void createRequestMap(const Ptree& config);
    void createMboxMap(const Ptree& config);
    void createRawHeadersList(const Ptree& config);
    set<string> createOauthScope(const Ptree& config) const;

    ClientIpHints getIpHints(const string& clientIp) const;
    size_t getStatusCacheTtl(const string& clientIp) const;

    void imapFoldersEncoding(const string& e)
    {
        imapFoldersEncoding_ = e;
    }
    const string& imapFoldersEncoding(void) const
    {
        return imapFoldersEncoding_;
    }

    ServerSettings serverSettings;

    AppendSettings appendSettings;
    SearchSettings searchSettings;

    uint64_t maxDBChunkSize = 500;

    uint32_t idleUpdateTimeout = 100;

    bool enableBBBlock = true;

    Ptree languageOptionsXml;
    bool allowInboxSubfolders = false;

    bool dropFreshCounter = false;
    bool checkKarma = false;

    set<std::string> oauthScope;

    size_t maxDetailsLoadingChunk = 100000;

    bool moveExtension = true;
    size_t idleExtension = false;
    bool startTLSExtension = true;
    bool oauthExtension = true;
    bool notifyExtension = false;

    ReportSpamSettings reportSpam;
    ImapLogSettings logSettings;
    LoginSettings loginSettings;

    vector<std::string> rawHeadersList;
    std::string allowZombie;

    size_t maxBadCommands = 100;

    bool useSettingsRepository = true;
    bool xivaAtLogin = true;
    size_t authRetry = 2;
    size_t authRetryTimeout = 3000;

    std::string defaultDomain;

    IpHintsVector ipHints;

    size_t searchTreeMaxHeight = 100;

    std::size_t fetchChunkSize = 10000;

    std::optional<ThrottlingSettings> throttlingSettings;

protected:
    set<std::string> readDropSessionEvents(const Ptree& config) const;
    set<std::string> readStringSet(const Ptree& config, const string& group, const string& child)
        const;

    IpHintsVector readIpHints(const PtreeConstOpt& ipHintsXml) const;

private:
    string imapFoldersEncoding_;

    boost::unordered_map<string, string> request_map_;

    bool pgUidsAll = false;
    set<std::string> pgUids;
};

using ImapSettings = Settings;
typedef std::shared_ptr<Settings> SettingsPtr;
typedef std::shared_ptr<const Settings> SettingsCPtr;

} // namespace yimap
