#pragma once

#include <common/types.h>
#include <common/global_stats.h>
#include <yplatform/module.h>
#include <yplatform/util/unordered.h>

#include <atomic>

namespace yimap {

struct execution
{
    unsigned long long calls = 0;
    Duration duration = Duration::zero();
    unsigned long long status_success = 0;
    unsigned long long status_fail = 0;
    unsigned long long status_uncertain = 0;
};

class Stats : public yplatform::module_stats
{
public:
    typedef yplatform::iunordered<string, execution>::map stat_map_t;

    Stats()
        : sessions(0)
        , sessions_cumulative(0)
        , dropped_sessions_cumulative(0)
        , fetch_bytes_cumulative(0)
        , fetch_buffer_size(0)
        , append_bytes_cumulative(0)
        , append_buffer_size(0)
        , read_bytes_cumulative(0)
        , read_buffer_size(0)
        , write_bytes_cumulative(0)
        , write_buffer_size(0)
        , cmds(0)
        , backend_reqs(0)
        , logins_granted(0)
        , logins_rejected(0)
        , logins_not_enabled(0)
        , logins_bb_errors(0)
        , logins_api_errors(0)
        , logins_disabled(0)
    {
    }

    virtual ~Stats() = default;

    std::atomic_int sessions;
    std::atomic_uint64_t sessions_cumulative;
    std::atomic_int64_t dropped_sessions_cumulative;
    std::atomic_int64_t fetch_bytes_cumulative;
    std::atomic_int64_t fetch_buffer_size;
    std::atomic_int64_t append_bytes_cumulative;
    std::atomic_int64_t append_buffer_size;
    std::atomic_int64_t read_bytes_cumulative;
    std::atomic_int64_t read_buffer_size;
    std::atomic_int64_t write_bytes_cumulative;
    std::atomic_int64_t write_buffer_size;
    unsigned int cmds;
    unsigned int backend_reqs;
    unsigned int logins_granted;
    unsigned int logins_rejected;
    unsigned int logins_not_enabled;
    unsigned int logins_bb_errors;
    unsigned int logins_api_errors;
    unsigned int logins_disabled;
    stat_map_t commands;
    stat_map_t api_calls;

protected:
    virtual ptree_ptr core_get_stat() const
    {
        ptree_ptr result = yplatform::module_stats::core_get_stat();
        result->put("sessions", sessions);
        result->put("sessions_cumulative", sessions_cumulative);
        result->put("dropped_sessions_cumulative", dropped_sessions_cumulative);
        result->put("fetch_bytes_cumulative", fetch_bytes_cumulative);
        result->put("fetch_buffer_size", fetch_buffer_size);
        result->put("append_bytes_cumulative", append_bytes_cumulative);
        result->put("append_buffer_size", append_buffer_size);
        result->put("read_bytes_cumulative", read_bytes_cumulative);
        result->put("read_buffer_size", read_buffer_size);
        result->put("write_bytes_cumulative", write_bytes_cumulative);
        result->put("write_buffer_size", write_buffer_size);
        result->put("cached_messages_count", GlobalStats::cachedMessagesCount);
        result->put("cmds", cmds);
        result->put("backend_requests", backend_reqs);
        result->put("logins.granted", logins_granted);
        result->put("logins.rejected", logins_rejected);
        result->put("logins.not_enabled", logins_not_enabled);
        result->put("logins.bb_errors", logins_bb_errors);
        result->put("logins.api_errors", logins_api_errors);
        result->put("logins.disabled", logins_disabled);
        for (stat_map_t::const_iterator i = commands.begin(), i_end = commands.end(); i != i_end;
             ++i)
        {
            string path = "commands." + i->first + ".";
            result->put(path + "calls", i->second.calls);
            result->put(path + "duration", toString(durationCast<Seconds>(i->second.duration)));
            result->put(path + "status_success", i->second.status_success);
            result->put(path + "status_fail", i->second.status_fail);
            result->put(path + "status_fail", i->second.status_uncertain);
        }
        for (stat_map_t::const_iterator i = api_calls.begin(), i_end = api_calls.end(); i != i_end;
             ++i)
        {
            string path = "api_calls." + i->first + ".";
            result->put(path + "calls", i->second.calls);
            result->put(path + "duration", toString(durationCast<Seconds>(i->second.duration)));
            result->put(path + "status_success", i->second.status_success);
            result->put(path + "status_fail", i->second.status_fail);
            result->put(path + "status_fail", i->second.status_uncertain);
        }
        return result;
    }
};

typedef boost::shared_ptr<Stats> StatsPtr;

}
