#include <common/uid_map.h>
#include <boost/bind.hpp>

namespace yimap {

void UidMap::insert(const MessageData& msg)
{
    MessageSet::iterator found = messages.find(msg);
    if (found == messages.end())
    {
        messages.insert(msg);
        return;
    }

    MessageData replacement;
    // In case of zero modseq, we should updadate only details
    if (msg.modseq == 0 && msg.details)
    {
        replacement = *found;
        replacement.details = msg.details;
        if (msg.time != 0)
        {
            replacement.time = msg.time;
        }
    }
    else
    {
        replacement = msg;
    }

    messages.erase(msg);
    messages.insert(replacement);
}

MessageData UidMap::pop()
{
    if (messages.empty()) throw std::runtime_error("UidMap::pop(): no messages");
    MessageSet::iterator first = messages.begin();
    MessageData result(*first);
    messages.erase(first);
    return result;
}

UidMapPtr UidMap::split(uint32_t count)
{
    UidMapPtr result(new UidMap());
    uint32_t processed = 0;
    while (processed < count && !messages.empty())
    {
        MessageSet::iterator first = messages.begin();
        result->insert(*first);
        messages.erase(first);
    }
    return result;
}

void UidMap::remove(uint32_t uid)
{
    messages.erase(MessageData(uid));
}

UidMapPtr UidMap::filterBySequence(const seq_range& seq)
{
    UidMapPtr result(new UidMap());
    for (const MessageData& msg : messages)
    {
        if (seq.contains(msg)) result->insert(msg);
    }
    return result;
}

UidMapPtr UidMap::filterMessages(boost::function<bool(const MessageData&)> match, bool allowPartial)
    const
{
    UidMapPtr result(new UidMap());
    for (const MessageData& msg : messages)
    {
        if (!msg.details && !allowPartial) continue;
        if (match(msg)) result->insert(msg);
    }
    return result;
}

bool flagsMatch(const MessageData& msg, const std::set<string>* set, const std::set<string>* unset)
{
    return msg.flags.hasAllFlags(*set) && !msg.flags.hasAtLeastOneFlag(*unset);
}

UidMapPtr UidMap::filterByFlags(const std::set<string>& set, const std::set<string>& unset)
{
    return filterMessages(boost::bind(flagsMatch, _1, &set, &unset), true);
}

UidMapPtr UidMap::intersect(UidMapPtr other)
{
    UidMapPtr result(new UidMap());

    MessageSet::iterator head1 = messages.begin();
    MessageSet::iterator tail1 = messages.end();
    MessageSet::iterator head2 = other->messages.begin();
    MessageSet::iterator tail2 = other->messages.end();

    while (head1 != tail1 && head2 != tail2)
    {
        if (head1->uid == head2->uid)
        {
            result->insert(*head1);
            ++head1;
            ++head2;
        }
        else if (head1->uid > head2->uid)
        {
            ++head2;
        }
        else
        {
            ++head1;
        }
    }
    return result;
}

UidMapPtr UidMap::without(UidMapPtr other)
{
    UidMapPtr result(new UidMap());

    MessageSet::iterator head = messages.begin();
    MessageSet::iterator tail = messages.end();
    MessageSet::iterator headOther = other->messages.begin();
    MessageSet::iterator tailOther = other->messages.end();

    while (head != tail || headOther != tailOther)
    {
        if (head == tail) break;
        if (headOther == tailOther)
        {
            result->insert(*head);
            head++;
            continue;
        }
        if (head->uid == headOther->uid)
        {
            ++head;
            ++headOther;
            continue;
        }
        if (head->uid > headOther->uid)
        {
            ++headOther;
        }
        else
        {
            result->insert(*head);
            ++head;
        }
    }
    return result;
}

UidMapPtr UidMap::with(UidMapPtr other)
{
    UidMapPtr result(new UidMap());

    MessageSet::iterator head = messages.begin();
    MessageSet::iterator tail = messages.end();
    MessageSet::iterator headOther = other->messages.begin();
    MessageSet::iterator tailOther = other->messages.end();

    while (head != tail || headOther != tailOther)
    {
        if (head == tail)
        {
            result->insert(*headOther);
            headOther++;
            continue;
        }
        if (headOther == tailOther)
        {
            result->insert(*head);
            head++;
            continue;
        }
        if (head->uid == headOther->uid)
        {
            result->insert(*head);
            ++head;
            ++headOther;
            continue;
        }
        if (head->uid > headOther->uid)
        {
            result->insert(*headOther);
            ++headOther;
        }
        else
        {
            result->insert(*head);
            ++head;
        }
    }
    return result;
}

range_t UidMap::uidRange() const
{
    if (empty()) return range_t(0, 0);
    uint32_t firstUid = messages.begin()->uid;
    uint32_t lastUid = messages.rbegin()->uid;
    return range_t(firstUid, lastUid);
}

UidSetPtr UidMap::toUidSet() const
{
    UidSetPtr resultSet(new UidSet());
    for (const MessageData& msg : messages)
    {
        resultSet->insert(msg.uid);
    }
    return resultSet;
}

UidSequence UidMap::toUidSequence() const
{
    UidSequence res;
    for (const MessageData& msg : messages)
    {
        res.push(msg.uid);
    }
    return res;
}

void UidMap::iterate(std::function<void(const MessageData&, uint32_t index)> callback) const
{
    uint32_t index = 0;
    for (auto& msg : messages)
    {
        callback(msg, index);
        index++;
    }
}

MidSet UidMap::asMidSet() const
{
    MidSet result;
    iterate([&result](const MessageData& m, uint32_t) { result.insert(m.mid); });
    return result;
}

void MidsCollector(const MessageData& message, uint32_t, MidList* result)
{
    if (message.mid != 0) result->push_back(message.mid);
}

MessageVector UidMap::toMessageVector() const
{
    MessageVector ret;
    ret.reserve(messages.size());
    for (auto& message : messages)
    {
        ret.emplace_back(message);
    }
    return ret;
}

MidListPtr UidMap::toMidList() const
{
    MidListPtr mids(new MidList());
    iterate(std::bind(MidsCollector, std::placeholders::_1, std::placeholders::_2, mids.get()));
    return mids;
}

void SmidsCollector(const MessageData& message, uint32_t, SmidList* result)
{
    string smid = message.smid();
    if (!smid.empty()) result->push_back(smid);
}

SmidListPtr UidMap::toSmidList() const
{
    SmidListPtr smids(new SmidList());
    iterate(std::bind(SmidsCollector, std::placeholders::_1, std::placeholders::_2, smids.get()));
    return smids;
}

uint32_t UidMap::updateBaseUid()
{
    if (messages.empty())
    {
        throw std::runtime_error("updateBaseUid: no messages");
    }

    auto iter = messages.begin();
    uint32_t newBaseUid = iter->uid;
    uint32_t diffOffset = iter->offset;

    for (auto& msg : messages)
    {
        msg.baseUid = newBaseUid;
        msg.offset -= diffOffset;
    }

    return newBaseUid;
}

std::string UidMap::dump() const
{
    std::ostringstream os;

    for (auto msg : messages)
    {
        os << msg.dump() << std::endl;
    }

    return os.str();
}

struct DefaultItemWriter
{
    template <class ItemType>
    std::ostream& operator()(const ItemType& item, std::ostream& os) const
    {
        os << item;
        return os;
    }
};

struct UidWriter
{
    std::ostream& operator()(const MessageData& item, std::ostream& os) const
    {
        os << item.uid;
        return os;
    }
};

template <class Collection, class ItemWriter>
std::ostream& writeToStream(std::ostream& os, const Collection& items, const ItemWriter& writeOne)
{
    size_t outputTail = 5;
    size_t total = items.size();

    os << "<";
    size_t i = 0;
    for (auto item : items)
    {
        if (i > outputTail && i < total - outputTail)
        {
            if (i == outputTail + 1)
            {
                os << "... " << total << " total ...";
            }
        }
        else
        {
            writeOne(item, os) << " ";
        }
        i++;
    }
    os << ">";
    return os;
}

std::ostream& operator<<(std::ostream& os, const UidMap& uidMap)
{
    return writeToStream(os, uidMap.messages, UidWriter());
}

std::ostream& operator<<(std::ostream& os, const SmidList& mids)
{
    return writeToStream(os, mids, DefaultItemWriter());
}

}
