#pragma once

#include <common/types.h>
#include <common/message_data.h>
#include <common/message_set.h>
#include <common/seq_range.h>
#include <common/uid_sequence.h>
#include <boost/shared_ptr.hpp>
#include <boost/optional.hpp>

#include <set>
#include <vector>

namespace yimap {

typedef std::set<uint32_t> UidSet;
typedef boost::shared_ptr<UidSet> UidSetPtr;

typedef std::vector<uint64_t> UidVector;
typedef boost::shared_ptr<UidVector> UidVectorPtr;

struct MidList : public std::vector<uint64_t>
{
};
typedef std::shared_ptr<MidList> MidListPtr;

class UidMap;
typedef std::shared_ptr<UidMap> UidMapPtr;

class UidMap
{
public:
    using Iterator = MessageSet::iterator;

    UidMap() = default;
    UidMap(const UidMap& that) = default;

    auto begin() const
    {
        return messages.cbegin();
    }

    auto rbegin() const
    {
        return messages.crbegin();
    }

    auto end() const
    {
        return messages.cend();
    }

    auto rend() const
    {
        return messages.crend();
    }

    inline std::size_t size() const
    {
        return messages.size();
    }
    inline bool empty() const
    {
        return messages.empty();
    }

    void clear()
    {
        messages.clear();
    }

    void insert(const MessageData& element);

    UidMapPtr filterBySequence(const seq_range& seq);

    void iterate(std::function<void(const MessageData&, uint32_t index)> callback) const;

    UidMapPtr filterMessages(
        boost::function<bool(const MessageData&)> match,
        bool allowPartial = false) const;
    UidMapPtr filterByFlags(const std::set<string>& set, const std::set<string>& unset);

    UidMapPtr intersect(UidMapPtr other);
    UidMapPtr without(UidMapPtr other);
    UidMapPtr with(UidMapPtr other);

    MessageVector toMessageVector() const;
    MidListPtr toMidList() const;
    SmidListPtr toSmidList() const;
    template <typename MidsCollection>
    void copyMids(MidsCollection&) const;

    UidSetPtr toUidSet() const;
    UidSequence toUidSequence() const;

    MidSet asMidSet() const;

    range_t uidRange() const;

    void remove(uint32_t uid);

    UidMapPtr split(uint32_t count);
    MessageData pop();

    uint32_t updateBaseUid();

    std::string dump() const;

    friend std::ostream& operator<<(std::ostream& os, const UidMap& uidMap);

private:
    MessageSet messages;
};

extern std::ostream& operator<<(std::ostream& os, const UidMap& uidMap);
extern std::ostream& operator<<(std::ostream& os, const SmidList& mids);

template <typename MidsCollection>
void UidMap::copyMids(MidsCollection& dest) const
{
    for (auto& msg : messages)
    {
        string smid = msg.smid();
        if (!smid.empty()) dest.push_back(smid);
    }
}

inline UidMapPtr clone(UidMapPtr source)
{
    return std::make_shared<UidMap>(*source);
}

} // namespace yimap
