#include "protocol_coroutine.h"
#include <network/session.h>
#include <backend/settings.h>
#include <backend/backend.h>
#include <backend/backend_service.h>
#include <backend/auth_ymodbb/auth_backend_ymodbb.h>
#include <backend/meta_pg/pg_backend.h>
#include <backend/mbody/mbody_backend.h>
#include <backend/xiva/module.h>
#include <backend/msearch.h>
#include <backend/append/append_backend.h>
#include <common/settings.h>
#include <common/imap_context.h>
#include <common/stats.h>
#include <common/log/log.h>
#include <yplatform/net/server.h>
#include <yplatform/module.h>

namespace yimap {

class ServerModule : public yplatform::module
{
public:
    ServerModule(yplatform::reactor& reactor, const Ptree& conf)
        : reactor_(yplatform::reactor::make_not_owning_copy(reactor))
    {
        if (!reactor_->plain())
        {
            throw std::runtime_error("imap server module requires "
                                     "a single-thread reactor - set pool_count=N and io_threads=1");
        }

        settings_ = std::make_shared<Settings>(conf);
        initIOData();
        initTCPServers();
        startLogs(conf);
    }

    void reload(const Ptree& /*conf*/)
    {
        rotateLogs();
    }

    void start()
    {
        size_t i = 0;
        for ([[maybe_unused]] auto& io : ioData_)
        {
            for (auto& ep : settings_->serverSettings.endpoints)
            {
                auto& tcpServer = tcpServers_[i++];
                startOneTCPServer(tcpServer, ep);
            }
        }
    }

    void stop()
    {
        for (auto& server : tcpServers_)
        {
            server.stop();
        }
        stopLogs();
    }

    const yplatform::module_stats_ptr get_module_stats() const override
    {
        return boost::static_pointer_cast<yplatform::module_stats>(stats_);
    }

    virtual StatsPtr stats()
    {
        return stats_;
    }

private:
    void initIOData()
    {
        for (size_t i = 0; i < reactor_->size(); ++i)
        {
            ioData_.emplace_back(*(*reactor_)[i]->io());
            ioData_.back().setup_ssl(settings_->serverSettings.ssl);
            ioData_.back().setup_dns();
        }
    }

    void initTCPServers()
    {
        // Read certificates and bind to sockets at init because
        // yplatform process may have insufficient privileges at start.
        for (size_t i = 0; i < ioData_.size(); ++i)
        {
            auto& io = ioData_[i];
            for (auto& ep : settings_->serverSettings.endpoints)
            {
                tcpServers_.emplace_back(io, ep.addr, ep.port, ep.socket_settings);
            }
        }
    }

    void startOneTCPServer(yplatform::net::tcp_server& tcp_server, const ServerEndpointSettings& ep)
    {
        YLOG_L(info) << "listen " << ep.addr << ":" << ep.port << " secure=" << ep.secure;
        // Where is no need to capture shared_from_this because of
        // yplatform modules and reactor destroy order guarantees.
        tcp_server.listen([this, &ep](auto&& socket) mutable {
            auto context = createContext(socket);
            auto backends = createBackendsBundle(context);
            auto session = boost::make_shared<server::TCPSession>(std::move(socket), ep, context);
            auto&& executor = context->ioService.get_executor();
            ProtocolCoroutine coro = { {}, context, session, backends };
            boost::asio::post(executor, coro);
        });
    }

    ImapContextPtr createContext(server::TCPSession::Socket& socket)
    {
        auto context = boost::make_shared<ImapContext>(*socket.get_io());
        context->settings = settings_;
        context->stats = stats_;
        context->sessionInfo = SessionInfo(
            socket.raw_socket().local_endpoint(), socket.raw_socket().remote_endpoint());
        auto clientIpHints = context->settings->getIpHints(context->sessionInfo.remoteAddress);
        context->sessionLogger.reset(
            createShortSessionInfo(*context), context->settings->logSettings, clientIpHints);
        return context;
    }

    backend::BundlePtr createBackendsBundle(ImapContextPtr context)
    {
        auto bundle = std::make_shared<backend::Bundle>();
        auto backendService = yplatform::find<backend::BackendService, std::shared_ptr>("mod_macs");
        auto pgBackend = std::make_shared<backend::PgBackend>(backendService, context);
        bundle->authBackend =
            std::make_shared<backend::AuthBackendYmodbb>(context, settings_->serverSettings.acfg);
        bundle->metaBackend = pgBackend;
        bundle->mbodyBackend = std::make_shared<mbody::MbodyBackendImpl>(context);
        if (settings_->useSettingsRepository)
        {
            bundle->settingsBackend = pgBackend;
        }
        else
        {
            bundle->settingsBackend = std::make_shared<backend::HTTPUserSettingsBackend>(context);
        }
        // FIXME no user data at start
        // journal = backendService->createJournal(
        //     context->userData.uid,
        //     context->sessionInfo.remoteAddress,
        //     context->userData.storage,
        //     context->userData.suid);
        bundle->journal = nullptr;
        auto controlServer = yplatform::find<backend::XivaModule>("xiva");
        bundle->notificationsBackend = controlServer->createBackend(context);
        bundle->search = std::make_shared<HTTPMSearchBackend>(context);
        bundle->append = std::make_shared<AppendBackendImpl>(backendService, context);
        return bundle;
    }

    yplatform::reactor_ptr reactor_;
    std::vector<yplatform::net::io_data> ioData_;
    std::vector<yplatform::net::tcp_server> tcpServers_;
    SettingsPtr settings_;
    StatsPtr stats_{ new Stats };
};

} // namespace yimap

#include <yplatform/module_registration.h>
DEFINE_SERVICE_OBJECT(yimap::ServerModule)
