#pragma once

#include <common/types.h>
#include <common/stats.h>
#include <yplatform/module.h>
#include <yplatform/util/unordered.h>

namespace yimap {

class ModuleContext : public yplatform::module_task_context
{
public:
    enum login_status
    {
        granted,
        rejected,
        not_enabled,
        bb_error,
        api_error,
        disabled,
        unknown
    };
    enum exec_status
    {
        success,
        fail,
        uncertain
    };

    ModuleContext() : cmds_(0), backend_reqs_(0)
    {
    }

    void setModuleStat(StatsPtr moduleStat)
    {
        moduleStat_ = moduleStat;
    }

    void stats(unsigned int cmds, unsigned int reqs, string const& cmd = string())
    {
        lock_t lock(mux());
        cmds_ += cmds;
        backend_reqs_ += reqs;
        if (!cmd.empty())
        {
            cmd_last_ = cmd;
            cmd_time_ = Clock::now();
        }
        {
            boost::unique_lock<boost::mutex> lck(moduleStat_->mux());
            moduleStat_->cmds += cmds;
            moduleStat_->backend_reqs += reqs;
        }
    }

    void statsLogin(login_status status)
    {
        boost::unique_lock<boost::mutex> lck(moduleStat_->mux());
        switch (status)
        {
        case granted:
            ++moduleStat_->logins_granted;
            break;
        case rejected:
            ++moduleStat_->logins_rejected;
            break;
        case not_enabled:
            ++moduleStat_->logins_not_enabled;
            break;
        case bb_error:
            ++moduleStat_->logins_bb_errors;
            break;
        case api_error:
            ++moduleStat_->logins_api_errors;
            break;
        case disabled:
            ++moduleStat_->logins_disabled;
            break;
        default:
            break;
        }
    }

    void stats_command(string const& cmd, Duration const& dur, exec_status status)
    {
        boost::unique_lock<boost::mutex> lck(moduleStat_->mux());
        execution& ex = moduleStat_->commands[cmd];

        ++ex.calls;
        ex.duration += dur;

        switch (status)
        {
        case success:
            ++ex.status_success;
            break;
        case fail:
            ++ex.status_fail;
            break;
        case uncertain:
            ++ex.status_uncertain;
            break;
        }
    }

    void stats_api(string const& name, Duration const& dur, exec_status status)
    {
        boost::unique_lock<boost::mutex> lck(moduleStat_->mux());
        execution& ex = moduleStat_->api_calls[name];

        ++ex.calls;
        ex.duration += dur;

        switch (status)
        {
        case success:
            ++ex.status_success;
            break;
        case fail:
            ++ex.status_fail;
            break;
        case uncertain:
            ++ex.status_uncertain;
            break;
        }
    }

protected:
    virtual ptree_ptr core_get_stat() const
    {
        ptree_ptr result = yplatform::module_task_context::core_get_stat();
        result->put("commands", cmds_);
        result->put("backend_requests", backend_reqs_);
        result->put("last_command", cmd_last_);
        result->put("last_command_start", cmd_time_.time_since_epoch());
        return result;
    }

private:
    unsigned int cmds_;
    unsigned int backend_reqs_;
    string cmd_last_;
    TimePoint cmd_time_;
    StatsPtr moduleStat_;
};

typedef boost::shared_ptr<ModuleContext> task_context_ptr;

}
