#include "session.h"
#include <common/imap_context.h>

#include <yplatform/net/handlers/timer_handler.h>
#include <yplatform/context_repository.h>
#include <yplatform/find.h>

namespace yimap { namespace server {

inline bool isCorrectEOF(const boost::system::error_code& e)
{
    // 'short read' error happens when work over ssl
    // it's not clear case so we just skip it
    static auto short_read = boost::asio::ssl::error::stream_truncated;
    return e == boost::asio::error::eof || e == short_read || e == boost::asio::error::shut_down ||
        e == boost::asio::error::broken_pipe || e == boost::asio::error::connection_reset;
}

TCPSession::TCPSession(
    Socket&& socket,
    const ServerEndpointSettings& endpoint,
    ImapContextPtr context)
    : socket_(std::move(socket))
    , settings_(context->settings->serverSettings)
    , endpoint_(endpoint)
    , context_(context)
    , readq_(boost::make_shared<Buffer>(
          settings_.min_read_buffer_chunk_size,
          settings_.max_read_buffer_chunk_size))
{
    setDefaultTimeouts();
    context_->stats->sessions_cumulative++;
    context_->stats->sessions++;
}

TCPSession::~TCPSession()
{
    context_->stats->sessions--;
    updateStatsOnReleaseReadBuffer(readq_->size());
    updateStatsOnReleaseWriteBuffer(writeBufferSize);
    std::ostringstream logoutStream;
    logoutStream << "TCPSession completed"
                 << " ssl=" << context_->sessionInfo.sslStatus << "["
                 << context_->sessionInfo.sslChipher << "]"
                 << " duration=" << context_->age() << " client_id=["
                 << context_->userData.clientId.raw << "]"
                 << " logout=" << context_->sessionState.wasLogout;

    getLogger().logString(logoutStream.str(), LogHelper::LH_LOG_SYSTEM);

    if (settings_.trafficLogEnabled)
    {
        flushTrafficLog();
    }
}

ClientStream TCPSession::clientStream()
{
    return ClientStream(shared_from(this), context_);
}

void TCPSession::sendClientStream(
    const yplatform::net::buffers::const_chunk_buffer& s,
    bool skip_log,
    const string& debugRecord)
{
    assert(ioService().get_executor().running_in_this_thread());

    // Don't accept data after shutdown or close.
    if (!isOpen()) return;

    if (writeBufferSize > settings_.max_write_buffer_size)
    {
        reportDropSession("write buffer overlow");
        shutdown();
        return;
    }

    updateStatsOnFillWriteBuffer(s.size());
    writeq_.push(s);

    if (!skip_log)
    {
        const char* buff_ptr = boost::asio::buffer_cast<const char*>(s);
        getLogger().logCommandResponse(
            boost::make_iterator_range(buff_ptr, buff_ptr + boost::asio::buffer_size(s)),
            debugRecord);
    }

    beginWrite();
}

void TCPSession::asyncStartTLS(ErrorCodeFunction hook)
{
    assert(ioService().get_executor().running_in_this_thread());

    if (context_->sessionInfo.tlsOn)
    {
        getLogger().logEvent() << "unexpected startTls, terminating session";
        return hook(boost::asio::error::shut_down); // TODO use own errc
    }

    assert(!isReading);

    isReading = true; // prevents starting new read operations from beginRead()
    socket_.async_tls_handshake(
        yplatform::net::tcp_socket::handshake_type::server,
        endpoint_.socket_settings.tls_timeout,
        [this, self = shared_from(this), hook](ErrorCode ec) {
            isReading = false;
            if (ec)
            {
                getLogger().logEvent() << "asyncStartTLS error: " << ec.message();
                return hook(ec);
            }
            context_->sessionInfo.tlsOn = true;
            context_->sessionInfo.trustedConnection = true;
            context_->sessionInfo.sslStatus = "yes";
            context_->sessionInfo.sslChipher = getSSLChiphers();
            // XXX
            context_->sessionLogger.reset(createShortSessionInfo(*context_));
            hook(ec);
        });
}

string TCPSession::getSSLChiphers()
{
    auto sslStream = socket_.stream().get_ssl_stream();
    if (!sslStream)
    {
        return "error: no ssl stream";
    }

    SSL* ssl = sslStream->native_handle();
    const SSL_CIPHER* cipher = SSL_get_current_cipher(ssl);
    int ssl_algbits = 0;
    int ssl_usebits = SSL_CIPHER_get_bits(cipher, &ssl_algbits);

    ostringstream chipherStream;
    chipherStream << SSL_get_version(ssl) << " with cipher " << SSL_CIPHER_get_name(cipher) << " ("
                  << ssl_usebits << "/" << ssl_algbits << " bits)";
    return chipherStream.str();
}

void TCPSession::beginWrite()
{
    if (isWriting) return;
    try
    {
        bool flush_ok = !writeq_.flush(settings_.limits.max_output_chunk);
        if (!flush_ok)
        {
            if (bgShutdown_)
            {
                socket_.close();
            }
            return;
        }
        isWriting = true;
        socket_.async_write(
            writeq_.send_queue(),
            endpoint_.socket_settings.write_timeout,
            std::bind(&TCPSession::handleWrite, shared_from(this), p::_1, p::_2));
    }
    catch (...)
    {
        isWriting = false;
        socket_.close();
        getLogger().logEvent() << "beginWrite exception";
    }
}

void TCPSession::handleWrite(const ErrorCode& e, size_t bytes)
{
    isWriting = false;

    if (e)
    {
        if (!isCorrectEOF(e))
        {
            getLogger().logEvent() << "write error: " << e.message();
        }
        if (e == boost::asio::error::operation_aborted && !isReading)
        {
            // TODO send autologout?
            socket_.close();
        }
        return;
    }

    updateStatsOnReleaseWriteBuffer(bytes);
    writeq_.consume(bytes);

    if (settings_.trafficLogEnabled)
    {
        writeTraffic += bytes;
        if (writeTraffic >= 2048)
        {
            flushTrafficLog();
        }
    }

    beginWrite();
}

void TCPSession::asyncRead(std::size_t atleast, ErrorCodeFunction hook)
{
    assert(ioService().get_executor().running_in_this_thread());

    assert(!isReading);

    try
    {
        isReading = true;
        socket_.async_read(
            // TODO settings.read_chunk_size
            readq_->prepare(atleast),
            endpoint_.socket_settings.read_timeout,
            std::bind(&TCPSession::handleRead, shared_from(this), p::_1, p::_2, hook),
            atleast);
    }
    catch (...)
    {
        isReading = false;
        socket_.close();
        getLogger().logEvent() << "asyncRead exception";
    }
}

void TCPSession::handleRead(const ErrorCode& e, std::size_t bytes, ErrorCodeFunction hook)
{
    isReading = false;

    if (e)
    {
        if (e != boost::asio::error::operation_aborted && !bgShutdown_)
        {
            if (!isCorrectEOF(e))
            {
                getLogger().logEvent() << "read error: " << e.message() << ", terminating session";
            }
            socket_.close();
        }
        return hook(e);
    }

    updateStatsOnFillReadBuffer(bytes);
    readq_->commit(bytes);

    hook(e);
}

TCPSession::BufferRange TCPSession::readBuffer()
{
    return boost::make_iterator_range(readq_->begin(), readq_->end());
}

TCPSession::Segment TCPSession::consumeReadBuffer(size_t bytes)
{
    updateStatsOnReleaseReadBuffer(bytes);
    return readq_->detach(readq_->begin() + bytes);
}

TCPSession::Segment TCPSession::consumeEntireReadBuffer()
{
    return consumeReadBuffer(readq_->size());
}

void TCPSession::cancelRunningOperations()
{
    socket_.cancel_operations();
}

void TCPSession::shutdown()
{
    assert(ioService().get_executor().running_in_this_thread());
    if (bgShutdown_) return;
    bgShutdown_ = true; // socket will be closed in beginWrite after all data is sent
    if (isWriting)
    {
        socket_.stream().shutdown(decltype(socket_)::raw_socket_t::shutdown_receive);
    }
    else
    {
        socket_.stream().shutdown(decltype(socket_)::raw_socket_t::shutdown_both);
        socket_.close();
    }
}

void TCPSession::flushTrafficLog()
{
    if (writeTraffic > 0)
    {
        getLogger().logTraffic("write " + std::to_string(writeTraffic) + " bytes");
        writeTraffic = 0;
    }
}

void TCPSession::updateStatsOnFillReadBuffer(size_t bytes)
{
    context_->stats->read_buffer_size += bytes;
    context_->stats->read_bytes_cumulative += bytes;
}

void TCPSession::updateStatsOnReleaseReadBuffer(size_t bytes)
{
    context_->stats->read_buffer_size -= bytes;
}

void TCPSession::updateStatsOnFillWriteBuffer(size_t bytes)
{
    writeBufferSize += bytes;
    context_->stats->write_buffer_size += bytes;
    context_->stats->write_bytes_cumulative += bytes;
}

void TCPSession::updateStatsOnReleaseWriteBuffer(size_t bytes)
{
    writeBufferSize -= bytes;
    context_->stats->write_buffer_size -= bytes;
}

void TCPSession::reportDropSession(const string& reason)
{
    getLogger().logString("drop session due to " + reason, LogHelper::LH_LOG_SYSTEM);
    ++context_->stats->dropped_sessions_cumulative;
}

void TCPSession::setDefaultTimeouts()
{
    assert(ioService().get_executor().running_in_this_thread());
    endpoint_.socket_settings.read_timeout = endpoint_.socket_settings.write_timeout =
        settings_.autologout;
}

void TCPSession::disableTimeouts()
{
    assert(ioService().get_executor().running_in_this_thread());
    endpoint_.socket_settings.read_timeout = endpoint_.socket_settings.write_timeout =
        Duration::max();
}

void TCPSession::setIdleTimeouts()
{
    assert(ioService().get_executor().running_in_this_thread());
    endpoint_.socket_settings.read_timeout = settings_.autologout_idle;
}

ImapContextPtr TCPSession::getContext()
{
    return context_;
}

SessionLogger& TCPSession::getLogger()
{
    return context_->sessionLogger;
}

} // namespace server
} // namespace yimap
