#pragma once

#include <common/settings.h>
#include <common/types.h>
#include <common/session_info.h>
#include <common/log/session_logger.h>
#include <common/zerocopy.h>
#include <network/client_stream.h>

#include <yplatform/net/buffer_sequence.h>
#include <yplatform/net/socket.h>
#include <yplatform/zerocopy/streambuf.h>

#include <boost/enable_shared_from_this.hpp>

namespace yimap {

class ImapContext;
typedef boost::shared_ptr<ImapContext> ImapContextPtr;

namespace server {

/*
Network connection wrapper.
- async recv data
- async send data in background
- graceful shutdown then all data is sent
*/
class Session
{
public:
    typedef zerocopy::BufferRange BufferRange;
    typedef zerocopy::Segment Segment;

    virtual ~Session() = default;

    virtual void asyncStartTLS(ErrorCodeFunction hook) = 0;
    virtual bool isOpen() const = 0;
    virtual void asyncRead(std::size_t atleast, ErrorCodeFunction handler) = 0;
    virtual BufferRange readBuffer() = 0;
    virtual Segment consumeReadBuffer(size_t bytes) = 0;
    virtual Segment consumeEntireReadBuffer() = 0;

    virtual ClientStream clientStream() = 0;
    virtual void sendClientStream(
        const yplatform::net::buffers::const_chunk_buffer& s,
        bool skip_log = false,
        const string& extraLogMessage = string{}) = 0;

    virtual void cancelRunningOperations() = 0;

    // Start graceful background shutdown.
    virtual void shutdown() = 0;

    virtual void setDefaultTimeouts() = 0;
    virtual void disableTimeouts() = 0;
    virtual void setIdleTimeouts() = 0;

    // Aux functions.
    virtual ImapContextPtr getContext() = 0;
    virtual const ServerEndpointSettings& endpoint() const = 0;
};

class TCPSession
    : public Session
    , public boost::enable_shared_from_this<TCPSession> // use std::
{
public:
    typedef zerocopy::Buffer Buffer;
    typedef zerocopy::BufferPtr BufferPtr;
    typedef yplatform::net::tcp_socket Socket;
    typedef yplatform::net::output_buffer_sequence<yplatform::net::buffers::const_chunk_buffer>
        output_queue_t;

    TCPSession(Socket&& socket, const ServerEndpointSettings& endpoint, ImapContextPtr context);
    ~TCPSession();

    void asyncStartTLS(ErrorCodeFunction hook) override;

    bool isOpen() const override
    {
        return socket_.is_open() && !bgShutdown_;
    }

    void asyncRead(std::size_t atleast, ErrorCodeFunction handler) override;
    BufferRange readBuffer() override;
    Segment consumeReadBuffer(size_t bytes) override;
    Segment consumeEntireReadBuffer() override;

    ClientStream clientStream() override;
    void sendClientStream(
        const yplatform::net::buffers::const_chunk_buffer& s,
        bool skip_log = false,
        const string& extraLogMessage = string{}) override;

    void cancelRunningOperations() override;

    void shutdown() override;

    void setDefaultTimeouts() override;
    void disableTimeouts() override; // TODO extract to processing looop
    void setIdleTimeouts() override; // TODO extract to IDLE command

    // Aux functions.
    ImapContextPtr getContext() override;
    const ServerEndpointSettings& endpoint() const override
    {
        return endpoint_;
    }

private:
    SessionLogger& getLogger();
    string getSSLChiphers();
    void beginWrite();
    void handleWrite(const ErrorCode& ec, size_t bytes);
    void handleRead(const ErrorCode& ec, std::size_t bytes, ErrorCodeFunction hook);
    void flushTrafficLog();
    void updateStatsOnFillReadBuffer(size_t bytes);
    void updateStatsOnReleaseReadBuffer(size_t bytes);
    void updateStatsOnFillWriteBuffer(size_t bytes);
    void updateStatsOnReleaseWriteBuffer(size_t bytes);
    void reportDropSession(const string& reason);

    IOService& ioService()
    {
        return *socket_.get_io();
    }

    Socket socket_;
    ServerSettings settings_;
    ServerEndpointSettings endpoint_;
    ImapContextPtr context_;

    BufferPtr readq_;
    output_queue_t writeq_;
    int64_t writeBufferSize = 0;

    size_t writeTraffic = 0;

    bool bgShutdown_ = false;
    bool isReading = false;
    bool isWriting = false;
};

typedef boost::shared_ptr<Session> SessionPtr;
typedef boost::weak_ptr<Session> SessionWeakPtr;

}

typedef server::Session NetworkSession;
typedef server::SessionPtr NetworkSessionPtr;
typedef server::SessionWeakPtr NetworkSessionWeakPtr;

}
