//#define BOOST_SPIRIT_DEBUG
#define BOOST_SPIRIT_THREADSAFE
#define PHOENIX_THREADSAFE

//#define DEBUG_AST 1

#include <parser/parser.h>

#include <yplatform/log.h>
#include <yplatform/util.h>

#include <parser/grammar.h>

namespace yimap { namespace parser {

class Module : public parser
{
public:
    void init(const Ptree& xml)
    {
        this->init_pool();
        unsigned nthreads = xml.get<unsigned>("threads");
        unsigned queue_capacity = xml.get("queue_capacity", 100);
        int enqueue_timeout = xml.get("enqueue_timeout_ms", -1);
        if (enqueue_timeout >= 0)
        {
            enqueue_timeout_ = boost::posix_time::milliseconds(enqueue_timeout);
        }

        L_(debug) << "activating parser::parser";
        this->set_capacity(queue_capacity);
        this->open(nthreads);
    }

    void fini(void)
    {
        close();
    }

protected:
    struct parser_context : public yplatform::active::object_context
    {
        ImapGrammar grammar;
    };

    parser_context* create_context(void) const override
    {
        std::auto_ptr<parser_context> ctx(new parser_context);
        return ctx.release();
    }

    Future<CommandASTPtr> parse(
        yplatform::task_context_ptr ctx,
        const zerocopy::SegmentRange& data,
        const yplatform::active::ptime& deadline) override
    {
        Promise<CommandASTPtr> prom;
        if (!this->enqueue_method(
                boost::bind(&Module::parse_i, this, ctx, prom, data, _1),
                boost::bind(&Module::deadline_cancel_hook, this, ctx, prom),
                1, // priority
                deadline))
        {
            TASK_LOG(ctx, error) << "Module::parse cancelled due to enqueue deadline";
            prom.set_exception(std::runtime_error("method cancelled due to enqueue deadline"));
        }

        return prom;
    }

    void parse_i(
        yplatform::task_context_ptr context,
        Promise<CommandASTPtr>& prom,
        const zerocopy::SegmentRange& data,
        yplatform::active::object_context* ctx);

    void deadline_cancel_hook(yplatform::task_context_ptr ctx, Promise<CommandASTPtr>& prom)
    {
        TASK_LOG(ctx, error) << "parser deadline_cancel_hook invoked";
        prom.set_exception(std::runtime_error("method cancelled due to deadline"));
    }

    const yplatform::active::time_duration get_enqueue_timeout() const override
    {
        return enqueue_timeout_;
    }

    yplatform::active::time_duration enqueue_timeout_;
};

#if DEBUG_AST
template <class TREES>
static void print_ast(yplatform::task_context_ptr ctx, const TREES& tree, int i = 0)
{
    for (const typename TREES::value_type& v : tree)
    {
        std::ostringstream os;
        os << "AST: " << std::string(i * 3, ' ');
        if (v.value.begin() == v.value.end()) os << "\"\"";
        else if (v.value.id().to_long() == lex_ids::PASSWORD)
        {
            os << "XXXXX";
        }
        else
        {
            yplatform::util::log_cmd(
                os, boost::make_iterator_range(v.value.begin(), v.value.end()), 1000);
        }
        os << "\t id=" << yimap::parser::lex_id_to_str(v.value.id().to_long());

        TASK_LOG(ctx, debug) << os.str();
        print_ast(ctx, v.children, i + 1);
    }
}
#endif

void Module::parse_i(
    yplatform::task_context_ptr context,
    Promise<CommandASTPtr>& prom,
    const zerocopy::SegmentRange& data,
    yplatform::active::object_context* ctx_ptr)
{
    parser_context& ctx = dynamic_cast<parser_context&>(*ctx_ptr);
    auto result = BSP::ast_parse<TreeFactory, TreeIterator>(
        data.begin(), data.end(), ctx.grammar, BSP::nothing_p);
    CommandASTPtr ptr(new CommandAST(std::move(result)));

#if DEBUG_AST
    print_ast(context, ptr->trees);
#endif

    prom.set(ptr);
}

}}

#include <yplatform/module_registration.h>
DEFINE_SERVICE_OBJECT(yimap::parser::Module)
