#define BOOST_SPIRIT_THREADSAFE
#define PHOENIX_THREADSAFE
// #define BOOST_SPIRIT_DEBUG
// #define _TEST_ 1

#include "rfc822.h"

#include <boost/algorithm/string.hpp>
#include <yplatform/util.h>
#include <yplatform/log.h>

#include <mimeparser/MessageParser.h>
#include <mimeparser/StringTools.h>

#include <iostream>
#include <fstream>

#if !defined(_TEST_)
#include <yplatform/log.h>
#endif

#include <boost/date_time/posix_time/posix_time.hpp>

namespace yimap { namespace rfc822 {
using namespace std;
using namespace yimap::rfc822;
using yplatform::iterator_range;
using yplatform::util::iequals;

template <class ForwardTraversalIterator>
class MyHeaderHandler
{
public:
    typedef ForwardTraversalIterator Iterator;
    typedef boost::iterator_range<ForwardTraversalIterator> Range;
    typedef MimeParser::HeaderData ParsedData;

public:
    MyHeaderHandler(MessageReactor& aReactor, const std::vector<std::string>& rawHeaderList)
        : m_isParsed(false), m_data(true), reactor(aReactor)
    {
        m_data.setRawHeaderList(rawHeaderList);
    }

    bool beginHeader(const Iterator& /*it*/)
    {
        return true;
    }
    bool endHeader(const Iterator& /*it*/)
    {
        m_isParsed = true;
        return true;
    }
    bool headerField(const Range& range, const Range& eol)
    {
        /// @note we can split that checks in functors.
        MimeParser::HeaderField headerField(range.begin(), eol.begin());
        if (!headerField.isValid())
        {
            return true;
        }
        std::string name = boost::to_lower_copy(headerField.name());
        if (name == "to" || name == "cc" || name == "bcc" || name == "from" || name == "sender" ||
            name == "reply_to")
        {
            return onFieldAddressList(headerField, range);
        }

        if (name[0] == 'c')
        {
            if (m_data.parseContentType(headerField))
            {
                onMimeContentType(headerField, range);
                return true;
            }
            if (m_data.parseContentDisposition(headerField))
            {
                onMimeField(
                    name, m_data.content_disposition(), "filename", m_data.filename(), range);
                return true;
            }
            if (m_data.parseContentTransferEncoding(headerField))
            {
                onMimeField(
                    name,
                    m_data.content_transfer_encoding(),
                    "",
                    mulca_mime::DecodedString(),
                    range);
                return true;
            }
            if (name == "content-language" || name == "content-location" || name == "content-id" ||
                name == "content-description" || name == "content-md5")
            {
                return parseMimeParam(headerField, range);
            }
        }
        onField(headerField.name(), headerField.value(), range);

        return true;
    }

    void onField(const string& cname, const string& cvalue, const Range& raw)
    {
        string name = cname;
        string fieldValue = cvalue;

        FieldData ctypeField;
        ctypeField.name = name;
        ctypeField.raw = string(raw.begin(), raw.end());

        FieldValuePtr value(new BaseFieldValue(fieldValue));
        ctypeField.value = value;
        reactor.on_field_data(ctypeField);
    }

    bool onFieldAddressList(const MimeParser::HeaderField& headerField, const Range& raw)
    {
        ::rfc2822::rfc2822address address(headerField.value());
        AddressListFieldValue* addressValue = new AddressListFieldValue();
        for (const ::rfc2822::address_pair_t& addr : address)
        {
            size_t atPos = addr.second.find('@');
            if (atPos == string::npos) continue;
            string local = addr.second.substr(0, atPos);
            string domain = addr.second.substr(atPos + 1);
            addressValue->addrs.push_back(AddressType(addr.first, local, domain));
        }

        FieldData mimeField;
        mimeField.raw = string(raw.begin(), raw.end());
        mimeField.name = headerField.name();
        mimeField.value.reset(addressValue);
        reactor.on_field_data(mimeField);
        return true;
    }

    void onMimeField(
        const string& name,
        const string& value,
        const string& paramName,
        const mulca_mime::DecodedString& param,
        const Range& raw)
    {
        mime_with_params_field_value* mimeFieldValue = new mime_with_params_field_value(value);
        if (!paramName.empty())
        {
            mimeFieldValue->params.push_back(mime_parameter(paramName, param.contents));
        }

        FieldData mimeField;
        mimeField.raw = string(raw.begin(), raw.end());
        mimeField.name = name;
        mimeField.value.reset(mimeFieldValue);
        reactor.on_field_data(mimeField);
    }

    void onMimeContentType(const MimeParser::HeaderField& headerField, const Range& raw)
    {
        mulca_mime::parameters_ng mimeParams(headerField.value());

        mime_content_type_field_value* mimeContentType = new mime_content_type_field_value();
        mimeContentType->content_type.type = mimeParams.token();
        mimeContentType->content_type.subtype = mimeParams.token2();

        if (iequals("multipart", mimeContentType->content_type.type))
        {
            if (mimeParams.exist("boundary"))
            {
                mimeContentType->params.push_back(
                    mime_parameter("boundary", mimeParams["boundary"].contents));
            }
        }
        if (mimeParams.exist("charset"))
        {
            mimeContentType->params.push_back(
                mime_parameter("charset", mimeParams["charset"].contents));
        }
        if (mimeParams.exist("name"))
        {
            mimeContentType->params.push_back(mime_parameter("name", mimeParams["name"].contents));
        }

        FieldData mimeField;
        mimeField.raw = string(raw.begin(), raw.end());
        mimeField.name = headerField.name();
        mimeField.value.reset(mimeContentType);
        reactor.on_field_data(mimeField);
    }

    bool parseMimeParam(const MimeParser::HeaderField& headerField, const Range& raw)
    {
        mulca_mime::parameters_ng mimeParams(headerField.value());

        typedef mulca_mime::parameters_ng::par_t par_t;
        const par_t& params = mimeParams();

        string token = MimeParser::trim(params.empty() ? headerField.value() : mimeParams.token());
        mime_with_params_field_value* mimeFieldValue = new mime_with_params_field_value(token);
        for (const par_t::value_type& pair : params)
        {
            mimeFieldValue->params.push_back(mime_parameter(pair.first, pair.second.contents));
        }

        FieldData mimeField;
        mimeField.raw = string(raw.begin(), raw.end());
        mimeField.name = headerField.name();
        mimeField.value.reset(mimeFieldValue);
        reactor.on_field_data(mimeField);
        return true;
    }

    bool isParsed() const
    {
        return m_isParsed;
    }
    const ParsedData& data() const
    {
        return m_data;
    }

private:
    bool m_isParsed;
    /// @note Maybe we need to implement it in functional way.
    ParsedData m_data;
    MessageReactor& reactor;
};

template <class ForwardTraversalIterator>
class MyHeaderHandlerFactory
{
public:
    typedef ForwardTraversalIterator Iterator;
    typedef MyHeaderHandler<Iterator> Handler;

public:
    MyHeaderHandlerFactory(MessageReactor& aReactor, const std::vector<std::string>& rawHeaderList)
        : m_doParseInline(false)
        , m_doCalculateRealLength(false)
        , reactor(aReactor)
        , rawHeaderList(rawHeaderList)
    {
    }
    std::auto_ptr<Handler> create()
    {
        return std::auto_ptr<Handler>(new Handler(reactor, rawHeaderList));
    }
    void setParseInline(bool doParseInline)
    {
        m_doParseInline = doParseInline;
    }
    /// @warning FIXME should be a param to parser
    bool doParseInline() const
    {
        return m_doParseInline;
    }
    void setCalculateRealLength(bool doCalculateRealLength)
    {
        m_doCalculateRealLength = doCalculateRealLength;
    }
    bool doCalculateRealLength() const
    {
        return m_doCalculateRealLength;
    }

private:
    bool m_doParseInline;
    bool m_doCalculateRealLength;
    MessageReactor& reactor;
    std::vector<std::string> rawHeaderList;
};

//=============================================================================

template <typename Iterator>
class MyMessageHandler
{
public:
    typedef boost::iterator_range<Iterator> Range;

public:
    MyMessageHandler(MessageReactor& aReactor) : m_partNumber(0), reactor(aReactor)
    {
    }
    bool beginMessage(const Iterator& /*position*/)
    {
        return true;
    }
    bool endMessage(const Iterator& position)
    {
        reactor.on_body(Range(bodyBegin, position));
        return true;
    }

    bool beginPart(const Iterator& position, const MyHeaderHandler<Iterator>&)
    {
        if (m_partNumber == 0)
        {
            bodyBegin = position;
        }
        ++m_partNumber;
        return true;
    }

    bool endPart(const Iterator& /* position */)
    {
        return true;
    }

    bool beginHeader(const Iterator& /* position */)
    {
        return true;
    }
    bool endHeader(const Iterator& position)
    {
        if (m_partNumber == 0)
        {
            headerEnd = position;
        }
        return true;
    }
    bool handleBodyLine(const Range& /* line */)
    {
        return true;
    }
    void addError(const std::string& error)
    {
        YLOG_G(error) << "MyMessageHandler error: " << error;
    }

    bool isOk()
    {
        return m_partNumber > 0;
    }

public:
    unsigned int partNumber()
    {
        return m_partNumber;
    }

private:
    Iterator headerEnd;
    Iterator bodyBegin;
    Iterator bodyEnd;
    unsigned int m_partNumber;
    MessageReactor& reactor;
};

bool parseMessage(const string& data, MessageReactor& reactor, const vector<string>& rawHeadersList)
{
    typedef string::const_iterator IteratorT;
    MyMessageHandler<IteratorT> messageHandler(reactor);
    MyHeaderHandlerFactory<IteratorT> headerHandlerFactory(reactor, rawHeadersList);

    MimeParser::Parsers::
        MessageParser<IteratorT, MyMessageHandler<IteratorT>, MyHeaderHandlerFactory<IteratorT>>
            messageParser(data.begin(), messageHandler, headerHandlerFactory);
    messageParser.push(data.end());
    messageParser.stop();

    messageHandler.isOk();
    return true;
}

//=============================================================================

}}

#if _TEST_

#include <iostream>
using namespace std;
using yplatform::iterator_range;
using namespace yimap::rfc822;

static const std::locale loc = std::locale();

struct TestActions : public MessageReactor
{
    virtual void on_field_data(const FieldData& fd)
    {
        cout << "*** got field data ***\n";

        cout << "field raw: >" << fd.raw << "<\n";
        cout << "*** field name: >" << fd.name << "<\n";

        {
            boost::shared_ptr<AddressListFieldValue> al =
                boost::dynamic_pointer_cast<AddressListFieldValue>(fd.value);
            if (al)
            {
                for (const AddressType& at : al->addrs)
                {
                    // cout << "address = * (" << at.name << ") + <" << at.local << " @ " <<
                    // at.domain << "> *\n";
                }

                return;
            }
        }

        {
            boost::shared_ptr<mime_content_type_field_value> mp =
                boost::dynamic_pointer_cast<mime_content_type_field_value>(fd.value);
            if (mp)
            {
                return;
            }
        }

        {
            if (iequals(fd.name, "content-type"))
            {
                cout << "!!! !!! could not parse content-type\n";
                cout << "*** *** field name: >" << fd.name << "<\n";
                cout << "--- --- header raw: >" << fd.raw << "<\n";
                cout << "--- --- field value raw: >" << fd.value->raw << "<\n";
            }
            boost::shared_ptr<mime_with_params_field_value> mp =
                boost::dynamic_pointer_cast<mime_with_params_field_value>(fd.value);
            if (mp)
            {
                return;
            }
        }

        {
            if (boost::istarts_with(fd.name, "content-", loc) || iequals(fd.name, "cc") ||
                iequals(fd.name, "to") || iequals(fd.name, "from") ||
                //          iequals (fd.name, "sender") ||
                iequals(fd.name, "bcc"))
            {
                cout << "!!! !!! could not parse mime header\n";
                cout << "*** *** field name: >" << fd.name << "<\n";
                cout << "--- --- header raw: >" << fd.raw << "<\n";
                cout << "--- --- field value raw: >" << fd.value->raw << "<\n";
            }
        }
    }

    virtual void on_body_prefix(const MessageReactor::data_range_t& data)
    {
        cout << "*** got body prefix: " << data << "\n";
    }

    virtual void on_body(const MessageReactor::data_range_t& data)
    {
        cout << "*** got body: " << data << "\n";
    }
};

int main()
{

    // ifstream fcin ("msg.txt");

    // typedef multi_pass<istreambuf_iterator<char> > multi_pass_iterator_t;
    typedef istream::char_type char_t;

    // multi_pass_iterator_t
    //    in_begin(boost::spirit::make_multi_pass(std::istreambuf_iterator<char_t>(cin))),
    //    in_end(boost::spirit::make_multi_pass(istreambuf_iterator<char_t>()));

    // rfc2822::printer_actions printer (cout, &has_body);
    // TestActions<multi_pass_iterator_t> actions;
    // ::yimap::rfc822::rfc2822::grammar<TestActions<multi_pass_iterator_t> > g (actions);

    {
        std::istreambuf_iterator<char> eos;
        std::istreambuf_iterator<char> iit(std::cin.rdbuf());
        string msg;

        while (iit != eos)
            msg += *iit++;

        TestActions actions;
        parseMessage(msg, actions);
    }

    // if (! parse (in_begin, in_end, g).full) {
    //   std::cout << "!!! !!! message parse failure\n";
    //   return EXIT_FAILURE;
    // }
    // else
    // {
    //   std::cout << "message parse ok\n";
    //   return EXIT_SUCCESS;
    // }
    return EXIT_SUCCESS;
}
#endif
